jQuery(document).ready(function($) {

    const $container = $('#chatbot-container');
    const $messages = $('#chatbot-messages');
    const $input = $('#chatbot-input');
    const $sendBtn = $('.chatbot-send-btn');
    const $typing = $('#chatbot-typing');
    const $subscriptionCardsContainer = $('#subscription-cards-container');
    const $logsCardsContainer = $('#logs-cards-container');
    const $modal = $('#startNewChatModal');
    const $btnWrapper = $('#chatbot-btn-wrapper');
    const root = document.documentElement;

    function detectIsDarkMode() {
        const savedMode = localStorage.getItem('chatbot-theme-mode');
        if (savedMode) {
            return savedMode === 'dark';
        }
        if (window.matchMedia) {
            const systemDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
            return systemDark;
        }
        return false;
    }

    function applyFallbackColors(isDarkMode) {
        const oldFallback = document.getElementById('chatbot-fallback-colors');
        if (oldFallback) oldFallback.remove();

        const rootEl = document.documentElement;
        const baseColor = getComputedStyle(rootEl).getPropertyValue('--wp--preset--color--base').trim();
        const contrastColor = getComputedStyle(rootEl).getPropertyValue('--wp--preset--color--contrast').trim();

        if (!baseColor || !contrastColor || baseColor === contrastColor) {
            const style = document.createElement('style');
            style.id = 'chatbot-fallback-colors';
            if (isDarkMode) {
                style.textContent = `
                .chatbot-header, .chatbot-footer, .chatbot-input-container, .message-content, .chatbot-input {
                    --wp--preset--color--base: #1a1a1a !important;
                    --wp--preset--color--contrast: #ffffff !important;
                    --wp--preset--color--secondary: #6c757d !important;
                    --wp--preset--color--tertiary: #2a2a2a !important;
                }`;
            } else {
                style.textContent = `
                .chatbot-header, .chatbot-footer, .chatbot-input-container, .message-content, .chatbot-input {
                    --wp--preset--color--base: #ffffff !important;
                    --wp--preset--color--contrast: #1a1a1a !important;
                    --wp--preset--color--secondary: #6c757d !important;
                    --wp--preset--color--tertiary: #f0f0f0 !important;
                }`;
            }
            document.head.appendChild(style);
        } else {
            console.log('[THEME DEBUG] WP preset colors detected — no fallback needed');
        }
    }

    function updateThemeToggleIcon(theme) {
        const $themeBtn = $('#chatbot-theme-toggle');

        if (!$themeBtn.length) {
            return;
        }

        if (theme === 'dark') {
            $themeBtn.html(`
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" viewBox="0 0 24 24">
                    <path d="M12 17.5a5.5 5.5 0 1 0 0-11 5.5 5.5 0 0 0 0 11zm0-13a1 1 0 0 0 1-1V2a1 1 0 0 0-2 0v1.5a1 1 0 0 0 1 1zm0 17a1 1 0 0 0-1 1V24a1 1 0 0 0 2 0v-1.5a1 1 0 0 0-1-1zM22 11h-1.5a1 1 0 0 0 0 2H22a1 1 0 0 0 0-2zM4.5 11H3a1 1 0 0 0 0 2h1.5a1 1 0 0 0 0-2zm13.44-6.56a1 1 0 0 0 0 1.41l1.06 1.06a1 1 0 0 0 1.41-1.41l-1.06-1.06a1 1 0 0 0-1.41 0zM5.05 17.09l-1.06 1.06a1 1 0 0 0 1.41 1.41l1.06-1.06a1 1 0 0 0-1.41-1.41zm13.44 0a1 1 0 0 0-1.41 1.41l1.06 1.06a1 1 0 0 0 1.41-1.41l-1.06-1.06zM6.51 6.51a1 1 0 0 0 0-1.41L5.45 4.04A1 1 0 0 0 4.04 5.45l1.06 1.06a1 1 0 0 0 1.41 0z"/>
                </svg>
            `).attr('title', 'Switch to light mode');
        } else {
            $themeBtn.html(`
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                    <path d="M6 .278a.768.768 0 0 1 .08.858 7.208 7.208 0 0 0-.878 3.46c0 4.021 3.278 7.277 7.318 7.277.527 0 1.04-.055 1.533-.16a.787.787 0 0 1 .81.316.733.733 0 0 1-.031.893A8.349 8.349 0 0 1 8.344 16C3.734 16 0 12.286 0 7.71 0 4.266 2.114 1.312 5.124.06A.752.752 0 0 1 6 .278z"/>
                </svg>
            `).attr('title', 'Switch to dark mode');
        }
    }

    function applyTheme(theme) {
        const isDark = theme === 'dark';
        $container.attr('data-theme', theme);
        $modal.attr('data-theme', theme);
        const $root = $('#mindweb-chatbot-root');
        if ($root.length) {
            $root.attr('data-theme', theme);
        } else {
            console.error('[THEME DEBUG]  #mindweb-chatbot-root NOT FOUND — dark mode CSS will NOT apply!');
        }

        localStorage.setItem('chatbot-theme-mode', theme);
        updateThemeToggleIcon(theme);
        applyFallbackColors(isDark);
    }

    function toggleTheme() {
        const currentTheme = localStorage.getItem('chatbot-theme-mode') || 'light';
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        applyTheme(newTheme);
    }

    const isDarkMode = detectIsDarkMode();
    const initialTheme = isDarkMode ? 'dark' : 'light';
    applyTheme(initialTheme);

    if (window.matchMedia) {
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
            if (!localStorage.getItem('chatbot-theme-mode')) {
                const newTheme = e.matches ? 'dark' : 'light';
                applyTheme(newTheme);
            } else {
                console.log('[THEME DEBUG] System theme changed but user has manual preference — ignoring');
            }
        });
    }

    const state = {
        historyLoaded: false,
        cachedHistory: null,
        paymentStarted: false,
        cancelSubscriptionId: null,
        reActiveSubId: null
    };

    const BADGE_TYPES = {
        0: '<span class="badge bg-secondary">Trial</span>',
        1: '<span class="badge bg-info">Paid</span>'
    };

    const BADGE_STATUS = {
        0: '<span class="badge bg-success">Active</span>',
        1: '<span class="badge bg-warning">Cancelled</span>',
        2: '<span class="badge bg-danger">Expired</span>'
    };

    function getTimeBasedGreeting() {
        const hour = new Date().getHours();
        const greetings = {
            morning: ["Good morning", "Morning", "Rise and shine", "Wishing you a great morning"],
            afternoon: ["Good afternoon", "Hope you're having a great day", "Enjoy your afternoon", "How's your day going"],
            evening: ["Good evening", "Hope your evening is going well", "Wishing you a pleasant evening", "Relax and enjoy your evening"],
            night: ["Good night", "Hope you're having a peaceful night", "Late night chat?", "Winding down for the night"]
        };
        const emojis = {
            morning: ["🌅", "☀️", "🌤️", "☕"],
            afternoon: ["☀️", "🌞", "💼", "✨"],
            evening: ["🌆", "🌇", "🌙", "🍵"],
            night: ["🌙", "🌌", "⭐", "😴"]
        };
        let period = "night";
        if (hour >= 5 && hour < 12) period = "morning";
        else if (hour >= 12 && hour < 17) period = "afternoon";
        else if (hour >= 17 && hour < 21) period = "evening";
        const greeting = greetings[period][Math.floor(Math.random() * greetings[period].length)];
        const emoji = emojis[period][Math.floor(Math.random() * emojis[period].length)];
        return { greeting, emoji };
    }

    function formatTimestamp(date = new Date()) {
        const hours = date.getHours();
        const minutes = date.getMinutes();
        const ampm = hours >= 12 ? 'PM' : 'AM';
        const displayHours = hours % 12 || 12;
        const displayMinutes = minutes < 10 ? '0' + minutes : minutes;
        return `${displayHours}:${displayMinutes} ${ampm}`;
    }

    function formatMyDate(date, locale = 'en-GB', options = {}) {
        const dateObj = date instanceof Date ? date : new Date(date);
        const defaultOptions = { year: 'numeric', month: '2-digit', day: '2-digit', hour: '2-digit', minute: '2-digit', second: '2-digit' };
        return new Intl.DateTimeFormat(locale, { ...defaultOptions, ...options }).format(dateObj);
    }

    function getInitialHistory() {
        return window.chatbotFrontend?.initialHistory || null;
    }

    function isValidSubscription(endDate, status) {
        if (!endDate) return false;
        try {
            const nowUTC = new Date();
            const endUTC = new Date(endDate);
            if (isNaN(endUTC.getTime())) return false;
            const isNotExpired = nowUTC <= endUTC;
            const isActiveOrCancelled = (status === 0 || status === 1);
            return isNotExpired && isActiveOrCancelled;
        } catch (error) {
            console.error('Error validating subscription:', error);
            return false;
        }
    }

    async function checkIndexedItems() {
        try {
            const response = await $.get(chatbotData.ajaxurl, { action: 'chatbot_check_indexed_items', app_guid: chatbotData.app_guid });
            if (response && response.success && response.data) return !!response.data.has_items;
            return false;
        } catch (error) {
            console.error('Error checking indexed items:', error);
            return false;
        }
    }

    async function checkTrialStatus() {
        try {
            const response = await $.get(chatbotData.ajaxurl, {
                action: 'chatbot_get_trial_status',
                nonce: chatbotData.nonce,
                app_guid: chatbotData.app_guid,
                stripe_base_url: chatbotData.stripe_base_url,
                product_id: chatbotData.product_id,
                status: chatbotData.trial_status
            });
            if (response.success && response.data) {
                const newEnd = response.data.newEndDate;
                const newStatus = response.data.status;
                const newType = response.data.type;
                if (newEnd) chatbotData.trialEndDate = newEnd;
                if (newStatus !== undefined && newStatus !== null) chatbotData.trial_status = newStatus;
                if (newType !== undefined && newType !== null) chatbotData.trial_type = newType;
            }
            return response;
        } catch (error) {
            console.error('AJAX Error:', error);
            throw error;
        }
    }

    async function addTrialWithoutUser() {
        try {
            const response = await 
            $.ajax({
                url: chatbotData.ajaxurl, 
                method: 'POST',
                data: { 
                    action: 'chatbot_add_trial_subscription_no_user', 
                    nonce: chatbotData.nonce, 
                    app_guid: chatbotData.app_guid
                 }, dataType: 'json' 
                });
            if (response.success) {
                $subscriptionCardsContainer.html(`<div class="success-message-card"><span class="dashicons dashicons-yes-alt"></span><p>Trial subscription added successfully! Loading your subscription...</p></div>`);
                setTimeout(() => loadSubscriptionsCards(), 1000);
            } else throw new Error(response.data?.message || 'Unknown error');
        } catch (error) {
            console.error('Error adding trial:', error);
            alert('Error adding trial: you have already used your trial, buy a new one.: ' + error.message);
        }
    }

    async function addTrial($btn, originalText) {
        try {
            const response = await 
            $.ajax({ 
                url: chatbotData.ajaxurl, 
                method: 'POST', 
                data: { 
                    action: 'chatbot_add_trial_subscription', 
                    nonce: chatbotData.nonce, 
                    app_guid: chatbotData.app_guid }, dataType: 'json' 
                });
            if (response.success) {
                $subscriptionCardsContainer.html(`<div class="success-message-card"><span class="dashicons dashicons-yes-alt"></span><p>Trial subscription added successfully! Loading your subscription...</p></div>`);
                setTimeout(() => loadSubscriptionsCards(), 1000);
            } else throw new Error(response.data?.message || 'Unknown error');
        } catch (error) {
            console.error('Error adding trial:', error);
            alert('Error adding trial: you have already used your trial, buy a new one.: ' + error.message);
            $btn.prop('disabled', false).html(originalText);
        }
    }

    $(document).on('click', '#add-trial-btn', function() {
        const $btn = $(this);
        const originalText = $btn.html();
        $btn.prop('disabled', true).html('<span class="spinner is-active" style="margin-right:5px; vertical-align:middle;"></span> Adding Trial...');
        addTrialWithoutUser();
    });

    async function cancelSubscription(subscriptionId) {
        try {
            const response = await 
            $.post(chatbotData.ajaxurl, { 
                action: 'chatbot_cancel_subscription', 
                nonce: chatbotData.nonce, 
                subscription_id: subscriptionId 
            });
            if (response.success) {
                ChatbotToast.success('Subscription auto-renew cancelled successfully.');
                setTimeout(() => loadSubscriptionsCards(), 100);
            } else throw new Error(response.data?.message || 'Unknown error');
        } catch (error) {
            ChatbotToast.error('Error cancelling subscription: ' + error.message);
            throw error;
        }
    }

    async function reactivateSubscription(subscriptionId) {
        try {
            const response = await 
            $.post(chatbotData.ajaxurl, { 
                action: 'chatbot_reactivate_subscription', 
                nonce: chatbotData.nonce, 
                subscription_id: subscriptionId 
            });
            if (response.success) {
                ChatbotToast.success('Subscription reactivated successfully.');
                setTimeout(() => loadSubscriptionsCards(), 100);
            } else throw new Error(response.data?.message || 'Unknown error');
        } catch (error) {
            ChatbotToast.error('Error reactivating subscription: ' + error.message);
            throw error;
        }
    }

    function loadChatHistory() {
        const historyData = getInitialHistory();
        if (historyData && state.historyLoaded) {
            displayHistory(historyData);
            $messages.scrollTop($messages[0].scrollHeight);
            restorePinnedMessages();
            return;
        }
        const currentPostId = chatbotFrontend.post_id || 0;
        $.get(`${chatbotFrontend.ajax_url}?action=chatbot_get_history&post_id=${currentPostId}`)
            .done(data => {
                if (data.success && Array.isArray(data.data) && data.data.length > 0) {
                    displayHistory(data.data);
                    state.historyLoaded = true;
                    state.cachedHistory = data.data;
                } else { state.historyLoaded = true; }
            })
            .fail(err => { console.error('Error loading history:', err); state.historyLoaded = true; });
    }

    function displayHistory(data) {
        if (!Array.isArray(data) || data.length === 0) return;
        $messages.empty();
        showDefaultMessage();
        let botBuffer = '';
        data.forEach((item, i) => {
            const isLast = i === data.length - 1;
            const nextIsUser = !isLast && data[i + 1].role === 'user';
            if (item.role === 'user') {
                if (botBuffer) { addMessage(botBuffer, 'bot', false); botBuffer = ''; }
                addMessage(item.message, 'user', false);
            } else if (item.role === 'bot') {
                botBuffer += item.message;
                if (isLast || nextIsUser) { addMessage(botBuffer, 'bot', false); botBuffer = ''; }
            }
        });
        restorePinnedMessages();
        $messages.scrollTop($messages[0].scrollHeight);
    }

    function showDefaultMessage() {
        const { greeting, emoji } = getTimeBasedGreeting();
        const timestamp = formatTimestamp();
        const $message = $(`
            <div class="chatbot-message bot-message default-welcome-message">
                <div class="message-avatar"><img src="${chatbotFrontend.bot_avatar}" class="chatbot-avatar-img" /></div>
                <div class="message-wrapper">
                    <div class="message-content"><strong>${emoji} ${greeting}!</strong><br>${chatbotFrontend.summary.replace(/\n/g, '<br>')}</div>
                    <span class="message-timestamp">${timestamp}</span>
                </div>
            </div>
        `);
        $messages.append($message);
    }

    function generateMessageId(text, type) {
        let hash = 0;
        const str = text + type;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return 'msg_' + Math.abs(hash);
    }

    function addMessage(text, type, scrollToBottom = true) {
        if (!text || text.trim() === '') return null;
        const avatar = type === 'bot' ? `<img src="${chatbotFrontend.bot_avatar}" class="chatbot-avatar-img" />` : `<img src="${chatbotFrontend.user_avatar}" class="chatbot-avatar-img" />`;
        const messageClass = type === 'bot' ? 'bot-message' : 'user-message';
        const time = formatTimestamp();
        const messageId = generateMessageId(text, type);
        const escapedText = text.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/\n/g, '<br>');
        const botRatingHtml = `
        <button class="copy-response" title="Copy"><svg xmlns="http://www.w3.org/2000/svg" aria-hidden="true" fill="none" viewBox="0 0 24 24" height="14" width="14"><path data-testid="copy-icon" fill="currentColor" d="M2 22h16V6H2v16ZM4 8h12v12H4V8Z"></path><path fill="currentColor" d="M6 2v2h14v14h2V2H6Z"></path></svg></button>
        <button class="pin-response" title="Pin your response">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20" class="pin-icon" fill="currentColor"><path d="M10.36 16.48l-2-3.16 1.12-6.12c0.04-0.24 0-0.48-0.16-0.68s-0.4-0.32-0.64-0.32h-6.8c-0.24 0-0.48 0.12-0.64 0.32s-0.24 0.44-0.16 0.68l1.16 6.12-2.08 3.24c-0.16 0.24-0.16 0.6-0.04 0.84 0.16 0.28 0.44 0.44 0.72 0.44h3.6v7.12c0 0.48 0.36 0.84 0.84 0.84s0.84-0.36 0.84-0.84v-7.12h3.6c0.48 0 0.84-0.36 0.84-0.84 0-0.2-0.12-0.4-0.2-0.52zM2.4 16.16l1.4-2.2c0.12-0.16 0.16-0.4 0.12-0.6l-1.04-5.48h4.84l-1.04 5.44c-0.04 0.2 0 0.44 0.12 0.6l1.4 2.2-5.8 0.04z"/></svg>
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="18" height="18" class="unpin-icon" fill="currentColor" style="display:none;"><path d="M20 14.274V15h-7v7l-.5 1-.5-1v-6.172L13.828 14h5.163c-.051-.65-.332-.856-1.333-1.515L16.93 12h-1.1l1.16-1.161.927.618c1.302.868 2.084 1.252 2.084 2.817zm2.4-10.966L3.307 22.399l-.707-.707L9.293 15H5v-.726c0-1.565.782-1.95 2.084-2.817l1.147-.765L10 4l-1.522-.43A2.029 2.029 0 0 1 7 1.619V1h11v.618a2.029 2.029 0 0 1-1.478 1.953L15 4l1.107 4.186L21.692 2.6zM10.137 3h4.724l1.388-.392A1.033 1.033 0 0 0 16.926 2H8.074a1.033 1.033 0 0 0 .676.608zm-.954 8h4.109l1.995-1.995L13.966 4h-2.931zm1.109 3l2-2H8.07l-.73.485c-1 .659-1.28.866-1.332 1.515z"/></svg>
        </button>`;
        const userCopyHtml = `<button class="copy-response" title="Copy"><svg xmlns="http://www.w3.org/2000/svg" aria-hidden="true" fill="none" viewBox="0 0 24 24" height="14" width="14"><path data-testid="copy-icon" fill="currentColor" d="M2 22h16V6H2v16ZM4 8h12v12H4V8Z"></path><path fill="currentColor" d="M6 2v2h14v14h2V2H6Z"></path></svg></button>`;
        const actionButtons = type === 'bot' ? botRatingHtml : userCopyHtml;
        const $message = $(`
            <div class="chatbot-message ${messageClass}" data-message-id="${messageId}">
                <div class="message-avatar">${avatar}</div>
                <div class="message-wrapper">
                    <div class="message-content">${escapedText}</div>
                    <div class="d-flex message-actions"><span class="message-timestamp">${time}</span>${actionButtons}</div>
                </div>
            </div>
        `);
        $messages.append($message);
        if (scrollToBottom) $messages.scrollTop($messages[0].scrollHeight);
        return $message;
    }

    $(document).on('click', '.pin-response', function() {
        const $btn = $(this);
        const $message = $btn.closest('.chatbot-message');
        const messageId = $message.attr('data-message-id');
        const isPinned = $message.hasClass('pinned');
        if (isPinned) {
            $message.removeClass('pinned');
            localStorage.removeItem('pinned_' + messageId);
            $btn.find('.pin-icon').show();
            $btn.find('.unpin-icon').hide();
        } else {
            $message.addClass('pinned');
            localStorage.setItem('pinned_' + messageId, '1');
            $btn.find('.pin-icon').hide();
            $btn.find('.unpin-icon').show();
        }
    });

    function restorePinnedMessages() {
        $('.chatbot-message').each(function() {
            const $message = $(this);
            const messageId = $message.attr('data-message-id');
            const isPinned = localStorage.getItem('pinned_' + messageId) === '1';
            if (isPinned) {
                $message.addClass('pinned');
                const $btn = $message.find('.pin-response');
                $btn.find('.pin-icon').hide();
                $btn.find('.unpin-icon').show();
            }
        });
    }

    const $checkbox = $('#chatbot-context-toggle');
    const savedState = localStorage.getItem('chatbot_ask_only_current_page');
    if (savedState !== null) $checkbox.prop('checked', savedState === '1');
    $checkbox.on('change', function() {
        localStorage.setItem('chatbot_ask_only_current_page', $(this).is(':checked') ? '1' : '0');
    });

    $(document).on('click', '.copy-response', function() {
        const $btn = $(this);
        const $message = $btn.closest('.chatbot-message');
        const $content = $message.find('.message-content');
        const htmlText = $content.html();
        const plainText = htmlText.replace(/<br\s*\/?>/gi, '\n').replace(/&amp;/g, '&').replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/&quot;/g, '"');
        navigator.clipboard.writeText(plainText).then(() => {
            const originalIcon = $btn.html();
            $btn.html(`<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" viewBox="0 0 16 16"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg>`);
            $btn.addClass('copied');
            setTimeout(() => { $btn.html(originalIcon); $btn.removeClass('copied'); }, 2000);
        }).catch(err => { console.error('Failed to copy:', err); alert('Failed to copy message'); });
    });

    async function sendMessage() {
        const message = $input.val().trim();
        const time = formatTimestamp();
        if (!message) return;
        addMessage(message, 'user');
        $input.val('').css('height', 'auto');
        $sendBtn.prop('disabled', true);
        $typing.show();
        const askOnlyCurrent = $('#chatbot-context-toggle').is(':checked');
        const formData = new URLSearchParams({
            action: 'chatbot_send_message_stream',
            nonce: chatbotFrontend.nonce,
            message: message,
            post_id: chatbotFrontend.post_id,
            screen_resolution: `${window.innerWidth}x${window.innerHeight}`,
            language_accepted: navigator.language,
            timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
            platform: navigator.platform,
            stream: 1,
            ask_only_current_page: askOnlyCurrent ? 1 : 0
        });
        try {
            const response = await fetch(chatbotFrontend.ajax_url, { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: formData.toString() });
            const reader = response.body.getReader();
            const decoder = new TextDecoder('utf-8');
            const $botMsg = $(`
                <div class="chatbot-message bot-message">
                    <div class="message-avatar"><img src="${chatbotFrontend.bot_avatar}" class="chatbot-avatar-img" /></div>
                    <div class="message-wrapper">
                        <div class="message-content"></div>
                        <div class="d-flex message-actions"><span class="message-timestamp">${time}</span></div>
                    </div>
                </div>
            `);
            $messages.append($botMsg);
            const $botContent = $botMsg.find('.message-content');
            let fullText = '';
            while (true) {
                const { value, done } = await reader.read();
                if (done) break;
                const chunk = decoder.decode(value, { stream: true });
                fullText += chunk;
                const escapedText = fullText.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/\n/g, '<br>');
                $botContent.html(escapedText);
                $messages.scrollTop($messages[0].scrollHeight);
            }
            const messageId = generateMessageId(fullText, 'bot');
            $botMsg.attr('data-message-id', messageId);
            const ratingHtml = `
            <button class="copy-response" title="Copy"><svg xmlns="http://www.w3.org/2000/svg" aria-hidden="true" fill="none" viewBox="0 0 24 24" height="14" width="14"><path data-testid="copy-icon" fill="currentColor" d="M2 22h16V6H2v16ZM4 8h12v12H4V8Z"></path><path fill="currentColor" d="M6 2v2h14v14h2V2H6Z"></path></svg></button>
            <button class="pin-response" title="Pin your response">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="18" class="pin-icon" fill="currentColor"><path d="M10.36 16.48l-2-3.16 1.12-6.12c0.04-0.24 0-0.48-0.16-0.68s-0.4-0.32-0.64-0.32h-6.8c-0.24 0-0.48 0.12-0.64 0.32s-0.24 0.44-0.16 0.68l1.16 6.12-2.08 3.24c-0.16 0.24-0.16 0.6-0.04 0.84 0.16 0.28 0.44 0.44 0.72 0.44h3.6v7.12c0 0.48 0.36 0.84 0.84 0.84s0.84-0.36 0.84-0.84v-7.12h3.6c0.48 0 0.84-0.36 0.84-0.84 0-0.2-0.12-0.4-0.2-0.52zM2.4 16.16l1.4-2.2c0.12-0.16 0.16-0.4 0.12-0.6l-1.04-5.48h4.84l-1.04 5.44c-0.04 0.2 0 0.44 0.12 0.6l1.4 2.2-5.8 0.04z"/></svg>
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="18" height="18" class="unpin-icon" fill="currentColor" style="display:none;"><path d="M20 14.274V15h-7v7l-.5 1-.5-1v-6.172L13.828 14h5.163c-.051-.65-.332-.856-1.333-1.515L16.93 12h-1.1l1.16-1.161.927.618c1.302.868 2.084 1.252 2.084 2.817zm2.4-10.966L3.307 22.399l-.707-.707L9.293 15H5v-.726c0-1.565.782-1.95 2.084-2.817l1.147-.765L10 4l-1.522-.43A2.029 2.029 0 0 1 7 1.619V1h11v.618a2.029 2.029 0 0 1-1.478 1.953L15 4l1.107 4.186L21.692 2.6zM10.137 3h4.724l1.388-.392A1.033 1.033 0 0 0 16.926 2H8.074a1.033 1.033 0 0 0 .676.608zm-.954 8h4.109l1.995-1.995L13.966 4h-2.931zm1.109 3l2-2H8.07l-.73.485c-1 .659-1.28.866-1.332 1.515z"/></svg>
            </button>`;
            $botMsg.find('.message-actions').append(ratingHtml);
            restorePinnedMessages();
        } catch (err) {
            console.error(err);
            addMessage('Error connecting to server.', 'bot');
        } finally {
            $sendBtn.prop('disabled', false);
            $typing.hide();
        }
    }

    function renderBuyButton(type, status) {
        const $wrapper = $('#btn-buy-wrapper');
        $('#refresh-plugin').hide();
        $wrapper.empty();
        $('#buy_subscription').hide();
        const subscriptionType = Number(type);
        const subscriptionStatus = Number(status);
        const shouldShowBuyButton = subscriptionType === 0 || subscriptionStatus === 2;
        if (shouldShowBuyButton) {
            const buyBtn = `<a href="https://subscriptions.smartprocesses.cloud/Product/${chatbotData.product_id}?app-guid=${chatbotData.app_guid}&customer-email=${chatbotData.user_email}" class="btn btn-success buy_subscription" id="buy_subscription" target="_blank"><i class="bi bi-bag"></i> <span>Buy</span></a>`;
            $wrapper.append(buyBtn).show();
        } else {
            $wrapper.empty();
            $('#buy_subscription').hide();
            $('#refresh-plugin').show();
        }
    }

    async function loadSubscriptionsCards() {
        $subscriptionCardsContainer.html(`<div class="loading-card d-flex justify-content-center"><span class="spinner is-active"></span><p>Loading subscriptions...</p></div>`);
        try {
            const response = await $.get(chatbotData.ajaxurl, { action: 'chatbot_get_all_subscriptions', nonce: chatbotData.nonce, product_id: chatbotData.product_id, app_guid: chatbotData.app_guid, stripe_base_url: chatbotData.stripe_base_url, status: chatbotData.trial_status, type: chatbotData.trial_type });
            $subscriptionCardsContainer.empty();
            if (response.success && Array.isArray(response.data) && response.data.length > 0) {
                response.data.forEach((item, index) => { $subscriptionCardsContainer.append(createSubscriptionCard(item, index)); });
            } else if (response.success && Array.isArray(response.data) && response.data.length === 0) {
                $subscriptionCardsContainer.append(`<div class="empty-state-card"><i class="bi bi-inbox"></i><h3>No Active Subscriptions</h3><p>Start your trial subscription to get access to all features</p><button id="add-trial-btn" class="btn-add-trial"><i class="bi bi-plus-circle"></i> Add Trial Subscription</button></div>`);
            }
        } catch (error) {
            console.error('Error loading subscriptions:', error);
            $subscriptionCardsContainer.html(`<div class="error-card"><i class="bi bi-exclamation-triangle"></i><p>Failed to load subscriptions. Please refresh the page.</p></div>`);
        }
    }

    function createSubscriptionCard(item, index) {
        const subscriptionId = `subscription-${index}`;
        let statusBadge = '', actionButton = '';
        if (item.type == 0) {
            if (item.status == 0) statusBadge = '<span class="badge-status active">ACTIVE</span>';
            else if (item.status == 1) statusBadge = '<span class="badge-status cancelled">CANCELLED</span>';
            else if (item.status == 2) statusBadge = '<span class="badge-status expired">EXPIRED</span>';
        } else if (item.type == 1) {
            if (item.status == 0) {
                statusBadge = '<span class="badge-status active">ACTIVE</span>';
                actionButton = `<button class="btn-cancel-subscription cancel_subscription" data-item-id="${item.stripeSubscriptionId}"><i class="bi bi-x-circle"></i> Cancel Auto-Renew</button>`;
            } else if (item.status == 1) {
                statusBadge = '<span class="badge-status cancelled">CANCELLED</span>';
                const now = new Date(), endDate = new Date(item.currentPeriodEnd);
                if (now <= endDate) actionButton = `<button class="btn-activate-subscription activate_subscription" data-item-id="${item.stripeSubscriptionId}"><i class="bi bi-arrow-repeat"></i> Activate Auto-Renew</button>`;
            } else if (item.status == 2) statusBadge = '<span class="badge-status expired">EXPIRED</span>';
        }
        const hasInvoices = item.invoices && item.invoices.length > 0;
        const invoiceCount = hasInvoices ? item.invoices.length : 0;
        return `<div class="subscription-card-wrapper"><div class="subscription-main-card" id="subscription-main-card"><div class="subscription-icon"><i class="bi bi-check-circle"></i></div><div class="subscription-info"><div class="subscription-header-row"><div class="subscription-type">${item.type == 0 ? 'Trial' : 'Paid'}</div>${statusBadge}</div><div class="subscription-dates"><div class="date-item"><span class="date-label">Activated At:</span><span class="date-value">${formatMyDate(item.createdAt)}</span></div><div class="date-separator">•</div><div class="date-item"><span class="date-label">Current Period Start:</span><span class="date-value">${formatMyDate(item.currentPeriodStart)}</span></div><div class="date-separator">•</div><div class="date-item"><span class="date-label">Current Period End:</span><span class="date-value">${formatMyDate(item.currentPeriodEnd)}</span></div></div>${(item.type === 1 && item.status === 0) ? `<div class="next-payment">Next Payment Date: ${formatMyDate(item.currentPeriodEnd)}</div>` : ''}</div><div class="subscription-price">${item.price ? `<div class="price-amount">€${item.price}</div>` : ''}</div><div class="subscription-actions">${actionButton}${hasInvoices ? `<button class="btn-toggle-invoices" data-target="${subscriptionId}"><i class="bi bi-chevron-down"></i> Invoices (${invoiceCount})</button>` : ''}</div></div>${hasInvoices ? `<div class="invoices-section" id="${subscriptionId}" style="display: none;">${item.invoices.map(invoice => `<div class="invoice-item"><div class="invoice-icon"><i class="bi bi-file-earmark-text"></i></div><div class="invoice-details"><div class="invoice-id">#${invoice.invoiceNumber || 'N/A'}</div><div class="invoice-meta">${formatMyDate(invoice.created)} • Package Price ${invoice.currency || '€'} • Tax Fee ${invoice.currency || '€'} • ${invoice.status || 'paid'}</div></div><button class="btn-download-invoice download-invoice" data-item-id="${invoice.invoiceNumber}"><i class="bi bi-download"></i> Download</button></div>`).join('')}</div>` : ''}</div>`;
    }

    $(document).on('click', '.btn-toggle-invoices', function() {
        const $btn = $(this), targetId = $btn.data('target'), $invoicesSection = $('#' + targetId), $icon = $btn.find('i');
        if ($invoicesSection.is(':visible')) { $invoicesSection.slideUp(300); $icon.removeClass('bi-chevron-up').addClass('bi-chevron-down'); }
        else { $invoicesSection.slideDown(300); $icon.removeClass('bi-chevron-down').addClass('bi-chevron-up'); }
    });

    $(document).on('click', '.download-invoice', function() {
        const invoiceNumber = $(this).data('item-id');
        ChatbotToast.info('Downloading invoice...');
        $.post(chatbotData.ajaxurl, { 
            action: 'chatbot_download_invoice', 
            nonce: chatbotData.nonce, 
            invoice_number: invoiceNumber 
        })
        .done(function(response) {
            if (response.success && response.data.url) {
                const link = document.createElement('a'); link.href = response.data.url; link.download = `Invoice-${invoiceNumber}.pdf`;
                document.body.appendChild(link); link.click(); document.body.removeChild(link);
                ChatbotToast.success('Invoice download started.');
            } else ChatbotToast.error('Failed to download invoice: ' + (response.data?.message || 'Unknown error'));
        })
        .fail(function(jqXHR, textStatus, errorThrown) { console.error('AJAX error:', textStatus, errorThrown); ChatbotToast.error('Failed to download invoice. Please try again later.'); });
    });

    async function loadLogsCards() {
        $logsCardsContainer.html(`<div class="loading-card d-flex justify-content-center"><span class="spinner is-active"></span><p>Loading logs...</p></div>`);
        try {
            const response = await $.get(chatbotData.ajaxurl, { action: 'chatbot_get_history_subscriptions', nonce: chatbotData.nonce });
            $logsCardsContainer.empty();
            if (response.success && Array.isArray(response.data) && response.data.length > 0) {
                response.data.forEach(log => { $logsCardsContainer.append(`<div class="log-card"><div class="log-card-icon"><i class="bi bi-clock-history"></i></div><div class="log-card-content"><div class="log-date">${formatMyDate(log.createdAt)}</div><p class="log-message">${log.description || 'No description'}</p></div></div>`); });
            } else { $logsCardsContainer.html(`<div class="empty-state-card"><i class="bi bi-inbox"></i><h3>No Logs Available</h3><p>Your subscription history will appear here</p></div>`); }
        } catch (error) { console.error('Error loading logs:', error); $logsCardsContainer.html(`<div class="error-card"><i class="bi bi-exclamation-triangle"></i><p>Failed to load logs. Please try again later.</p></div>`); }
    }

    async function handleSubscriptionRefresh() {
        state.paymentStarted = false;
        try {
            const response = await checkTrialStatus();
            const newType = Number(response.data?.type), newStatus = Number(response.data?.status), newEndDate = response.data?.newEndDate;
            chatbotData.trial_type = newType; chatbotData.trial_status = newStatus; chatbotData.trialEndDate = newEndDate;
            const hasItems = await checkIndexedItems();
            const isValid = isValidSubscription(newEndDate, newStatus);
            if (hasItems && isValid) $('#chatbot-btn-wrapper').show(); else $('#chatbot-btn-wrapper').hide();
            setTimeout(() => { renderBuyButton(newType, newStatus); loadSubscriptionsCards(); }, 500);
        } catch (error) { console.error('Error refreshing subscription:', error); }
    }

    async function initializeChat() {
        const $btnWrapper = $('#chatbot-btn-wrapper');
        if (!$btnWrapper.length) return;
        try {
            const [hasItems, trialResponse] = await Promise.all([checkIndexedItems(), checkTrialStatus()]);
            const newEndDate = trialResponse.data?.newEndDate || chatbotData.trialEndDate;
            const status = Number(trialResponse.data?.status ?? chatbotData.trial_status);
            chatbotData.trialEndDate = newEndDate; chatbotData.trial_status = status;
            if (hasItems && isValidSubscription(newEndDate, status)) $btnWrapper.addClass('visible'); else $btnWrapper.removeClass('visible');
        } catch (err) { console.error('Init error:', err); }
    }

    $(document).off('click', '#clear-chat-history').on('click', '#clear-chat-history', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $modal.addClass('show').css('display', 'flex');
    });

    $(document).off('click', '#startNewChatModal .btn-close').on('click', '#startNewChatModal .btn-close', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('[MODAL] Close button clicked');
        $modal.removeClass('show').css('display', 'none');
    });

    $(document).off('click', '#startNewChatModal').on('click', '#startNewChatModal', function(e) {
        if (e.target === this) {
            $modal.removeClass('show').css('display', 'none');
        }
    });

    $(document).off('click', '#startNewChatModal .modal-content').on('click', '#startNewChatModal .modal-content', function(e) {
        e.stopPropagation();
    });

    $(document).off('click', '#confirmNewChatBtn').on('click', '#confirmNewChatBtn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        if (!chatbotFrontend || !chatbotFrontend.nonce) {
            console.error('Nonce not found in chatbotFrontend');
            return;
        }
        
        $.post(chatbotFrontend.ajax_url, {
            action: 'chatbot_clear_history',
            nonce: chatbotFrontend.nonce
        })
        .done(data => {
            if (data.success) {
                $messages.empty();
                state.historyLoaded = false;
                state.cachedHistory = null;
                showDefaultMessage();
                $modal.removeClass('show').css('display', 'none');
                console.log('[MODAL] Chat history cleared successfully');
            } else {
                alert(data.data?.message || 'Failed to clear history');
                $modal.removeClass('show').css('display', 'none');
            }
        })
        .fail((xhr, status, error) => {
            console.error('AJAX error:', { status, error, response: xhr.responseText });
            alert('Something went wrong. Please try again.');
            $modal.removeClass('show').css('display', 'none');
        });
    });

    $('#open-chatbot').on('click', () => {
        $container.fadeIn(100);
        const savedTheme = localStorage.getItem('chatbot-theme-mode') || 'light';
        console.log('[THEME DEBUG] Chat opened — re-applying icon for theme:', savedTheme);
        updateThemeToggleIcon(savedTheme);
        if ($messages.children().length === 0) showDefaultMessage();
        if (!state.historyLoaded) loadChatHistory();
        $input.focus();
        $btnWrapper.hide();
    });

    const svgExpand = `<svg xmlns="http://www.w3.org/2000/svg" fill="#000000" width="64px" height="64px" viewBox="-16 -16 64 64" style="fill-rule:evenodd;clip-rule:evenodd;stroke-linejoin:round;stroke-miterlimit:2;"><path d="M7.539,26.475l6.849,-6.971c0.58,-0.591 0.572,-1.541 -0.019,-2.121c-0.591,-0.58 -1.541,-0.572 -2.121,0.019l-6.737,6.856c-0.007,-0.079 -0.011,-0.159 -0.011,-0.24c0,-0 -0,-7.018 -0,-7.018c-0,-0.828 -0.672,-1.5 -1.5,-1.5c-0.828,0 -1.5,0.672 -1.5,1.5l0,7.018c0,3.037 2.462,5.5 5.5,5.5c3.112,-0 6.905,-0 6.905,-0c0.828,-0 1.5,-0.673 1.5,-1.5c0,-0.828 -0.672,-1.5 -1.5,-1.5l-6.905,-0c-0.157,-0 -0.311,-0.015 -0.461,-0.043Z"/><path d="M24.267,5.51l-7.056,7.181c-0.58,0.591 -0.571,1.541 0.019,2.122c0.591,0.58 1.541,0.571 2.121,-0.019l7.149,-7.277c0.031,0.156 0.047,0.318 0.047,0.483c-0,0 -0,6.977 -0,6.977c-0,0.828 0.672,1.5 1.5,1.5c0.828,0 1.5,-0.672 1.5,-1.5l-0,-6.977c-0,-3.038 -2.463,-5.5 -5.5,-5.5c-3.162,0 -7.047,0 -7.047,0c-0.828,0 -1.5,0.672 -1.5,1.5c0,0.828 0.672,1.5 1.5,1.5c0,0 3.885,0 7.047,0c0.074,-0 0.147,0.003 0.22,0.01Z"/></svg>`;
    const svgCompress = `<svg xmlns="http://www.w3.org/2000/svg" fill="#000000" width="64px" height="64px" viewBox="-16 -16 64 64" style="fill-rule:evenodd;clip-rule:evenodd;stroke-linejoin:round;stroke-miterlimit:2;"><path d="M11.21,18.511l-8.284,8.433c-0.58,0.59 -0.572,1.541 0.019,2.121c0.59,0.58 1.541,0.572 2.121,-0.019l8.378,-8.527c0.03,0.156 0.046,0.317 0.046,0.482c0,0 0,6.977 0,6.977c0,0.828 0.672,1.5 1.5,1.5c0.828,0 1.5,-0.672 1.5,-1.5l0,-6.977c0,-3.037 -2.462,-5.5 -5.5,-5.5c-3.14,0 -6.984,0 -6.984,0c-0.828,0 -1.5,0.672 -1.5,1.5c-0,0.828 0.672,1.5 1.5,1.5l6.984,0c0.074,0 0.148,0.004 0.22,0.01Z"/><path d="M18.496,11.437c-0.012,-0.102 -0.019,-0.205 -0.019,-0.31c0,-0 0,-7.145 0,-7.145c0,-0.827 -0.672,-1.5 -1.5,-1.5c-0.828,0 -1.5,0.673 -1.5,1.5l0,7.145c0,3.038 2.463,5.5 5.5,5.5c3.139,0 6.981,0 6.981,0c0.828,0 1.5,-0.672 1.5,-1.5c-0,-0.828 -0.672,-1.5 -1.5,-1.5l-6.981,0c-0.135,0 -0.267,-0.011 -0.396,-0.031l8.421,-8.572c0.58,-0.591 0.572,-1.541 -0.019,-2.121c-0.59,-0.581 -1.541,-0.572 -2.121,0.018l-8.366,8.516Z"/></svg>`;

    $('#resize-chatbot').on('click', () => {
        if ($container.hasClass('maximized')) { 
            $container.removeClass('maximized').addClass('minimized'); 
            $modal.removeClass('maximized').addClass('minimized'); 
            $('#resize-chatbot').html(svgExpand); 
        }
        else { 
            $container.removeClass('minimized').addClass('maximized'); 
            $modal.removeClass('minimized').addClass('maximized'); 
            $('#resize-chatbot').html(svgCompress); 
        }
    });

    $('#close-chatbot').on('click', () => {
        $container.addClass('closing');
        setTimeout(() => { $container.hide(); $container.removeClass('closing'); $btnWrapper.show(); }, 500);
    });

    $(document).on('click', '#chatbot-theme-toggle', function() {
        toggleTheme();
    });

    $input.on('input', function() { this.style.height = 'auto'; this.style.height = this.scrollHeight + 'px'; });
    $input.on('keydown', function(e) { if (e.key === 'Enter' && !e.shiftKey) { e.preventDefault(); sendMessage(); } });
    $sendBtn.on('click', sendMessage);

    $(document).on('click', '#buy_subscription', function() {
        state.paymentStarted = true; 
        $(this).hide(); 
        $('#refresh-plugin').show(); 
        });
    $('#refresh-plugin').on('click', handleSubscriptionRefresh);
    $(document).on('click', '.cancel_subscription', function() {
        openSubscriptionModal('cancel', $(this).data('item-id')); 
    });
    $(document).on('click', '.activate_subscription', function() {
        openSubscriptionModal('reactivate', $(this).data('item-id')); 
    });

    $('#subscriptionActionModalConfirm').on('click', async function() {
        const subscriptionId = $(this).data('subscription-id');
        const action = $(this).data('action');
        const btn = $(`button[data-item-id="${subscriptionId}"]`);
        
        btn.prop('disabled', true).text('Processing...');
        
        try {
            if (action === 'cancel') await cancelSubscription(subscriptionId);
            else if (action === 'reactivate') await reactivateSubscription(subscriptionId);
        } catch (error) { 
            btn.prop('disabled', false).text(action === 'cancel' ? 'Cancel Auto-Renew' : 'Activate Auto-Renew'); 
        }
        finally { 
            $('#subscriptionActionModal').removeClass('show').css('display', 'none');
        }
    });

    $('#subscription-tab').on('click', async function() {
        const response = await checkTrialStatus();
        renderBuyButton(Number(response.data?.type), Number(response.data?.status));
        loadSubscriptionsCards();
    });

    function openSubscriptionModal(action, subscriptionId) {
        if (action === 'cancel') { 
            ChatbotModal.confirm('Cancel Auto-Renew', 'Are you sure you want to cancel auto-renew for this subscription?', () => cancelSubscription(subscriptionId), 'Confirm Cancel', 'btn-danger'); 
        }
        else if (action === 'reactivate') { 
            ChatbotModal.confirm('Re-activate', 'Are you sure you want to re-activate this subscription?', () => reactivateSubscription(subscriptionId), 'Re-activate Subscription', 'btn-success'); 
        }
    }

    $('#logs-tab').on('click', loadLogsCards);
    initializeChat();

    window.chatbotTheme = {
        toggle: toggleTheme,
        updateIcon: updateThemeToggleIcon,
        detect: detectIsDarkMode,
        applyFallback: applyFallbackColors
    };
});
