/**
 * @package Chatbot_Plugin
 * @version 1.0.0
 * 
 * @example
 * // Basic usage
 * const modal = new ChatbotModal('my-modal');
 * modal.show();
 * 
 * // With options
 * const modal = new ChatbotModal('my-modal', {
 *     backdrop: 'static',
 *     keyboard: false
 * });
 * 
 * // Confirmation modal
 * ChatbotModal.confirm('Delete Item', 'Are you sure?', () => {
 *     console.log('Confirmed!');
 * });
 */

class ChatbotModal {
    /**
     * @param {string} id
     * @param {object} options 
     */
    constructor(id, options = {}) {
        this.id = id;
        this.element = document.getElementById(id);
        this.modal = null;
        this.options = {
            backdrop: 'static',
            keyboard: true,
            focus: true,
            ...options
        };
        
        if (!this.element) {
            console.error(`Modal element with ID "${id}" not found`);
        }
    }
    
    /**
     * Show modal
     * @returns {ChatbotModal} Returns self for chaining
     */
    show() {
        if (!this.element) return this;
        
        if (!this.modal) {
            this.modal = new bootstrap.Modal(this.element, this.options);
        }
        
        this.modal.show();
        return this;
    }
    
    /**
     * Hide modal
     * @returns {ChatbotModal} Returns self for chaining
     */
    hide() {
        if (this.modal) {
            this.modal.hide();
        }
        return this;
    }
    
    /**
     * Toggle modal visibility
     * @returns {ChatbotModal} Returns self for chaining
     */
    toggle() {
        if (this.modal) {
            this.modal.toggle();
        } else {
            this.show();
        }
        return this;
    }
    
    /**
     * Set modal title
     * 
     * @param {string} title New title
     * @returns {ChatbotModal} Returns self for chaining
     */
    setTitle(title) {
        const titleEl = this.element?.querySelector('.modal-title');
        if (titleEl) {
            titleEl.textContent = title;
        }
        return this;
    }
    
    /**
     * Set modal content
     * 
     * @param {string} content HTML content
     * @returns {ChatbotModal} Returns self for chaining
     */
    setContent(content) {
        const body = this.element?.querySelector('.modal-body');
        if (body) {
            body.innerHTML = content;
        }
        return this;
    }
    
    /**
     * Set modal footer
     * 
     * @param {string} footer HTML footer content
     * @returns {ChatbotModal} Returns self for chaining
     */
    setFooter(footer) {
        let footerEl = this.element?.querySelector('.modal-footer');
        
        if (!footerEl && footer) {
            // Create footer if it doesn't exist
            footerEl = document.createElement('div');
            footerEl.className = 'modal-footer';
            this.element.querySelector('.modal-content').appendChild(footerEl);
        }
        
        if (footerEl) {
            footerEl.innerHTML = footer;
        }
        
        return this;
    }
    
    /**
     * Set loading state
     * 
     * @param {boolean} isLoading Show/hide loading state
     * @param {string} message Loading message
     * @returns {ChatbotModal} Returns self for chaining
     */
    setLoading(isLoading = true, message = 'Loading...') {
        const body = this.element?.querySelector('.modal-body');
        
        if (!body) return this;
        
        if (isLoading) {
            body.innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-3 mb-0">${message}</p>
                </div>
            `;
        }
        
        return this;
    }
    
    /**
     * Attach confirm button callback
     * 
     * @param {function} callback Function to call on confirm
     * @returns {ChatbotModal} Returns self for chaining
     */
    onConfirm(callback) {
        const confirmBtn = this.element?.querySelector('[data-confirm]');
        
        if (confirmBtn) {
            confirmBtn.onclick = () => {
                callback();
                this.hide();
            };
        }
        
        return this;
    }
    
    /**
     * Add event listener to modal
     * 
     * @param {string} event Event name (show.bs.modal, shown.bs.modal, hide.bs.modal, hidden.bs.modal)
     * @param {function} callback Event callback
     * @returns {ChatbotModal} Returns self for chaining
     */
    on(event, callback) {
        if (this.element) {
            this.element.addEventListener(event, callback);
        }
        return this;
    }
    
    /**
     * Dispose modal instance
     */
    dispose() {
        if (this.modal) {
            this.modal.dispose();
            this.modal = null;
        }
    }
    
    /**
     * Create and show a confirmation modal (static method)
     * 
     * @param {string} title Modal title
     * @param {string} message Confirmation message
     * @param {function} onConfirm Callback when confirmed
     * @param {string} confirmText Confirm button text (default: 'Confirm')
     * @param {string} confirmClass Confirm button class (default: 'btn-danger')
     * @returns {ChatbotModal} Returns modal instance
     */
   static confirm(title, message, onConfirm, confirmText = 'Unindex', confirmClass = 'btn-danger') {
    const existing = document.getElementById('chatbot-confirm-modal');
    if (existing) existing.remove();

    const modalHTML = `
        <div class="modal fade" id="chatbot-confirm-modal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered" style="max-width: 500px;">
                <div class="modal-content rounded-3 shadow">
                    <div class="modal-header text-white" style="background:linear-gradient(135deg, #1a365d 0%, #0f172a 100%) !important">
                        <h5 class="modal-title-header">${title}</h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body py-4">
                        <p class="mb-0">${message}</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="button" class="btn ${confirmClass}" data-confirm>
                            <i class="bi bi-trash"></i> ${confirmText}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    document.body.insertAdjacentHTML('beforeend', modalHTML);

    const modal = new ChatbotModal('chatbot-confirm-modal');
    modal.show().onConfirm(onConfirm);

    modal.on('hidden.bs.modal', () => {
        modal.element.remove();
    });

    return modal;
}

    
    /**
     * Create and show a loading modal (static method)
     * 
     * @param {string} title Modal title
     * @param {string} message Loading message
     * @returns {ChatbotModal} Returns modal instance
     */
    static loading(title, message = 'Loading...') {
        const existing = document.getElementById('chatbot-loading-modal');
        if (existing) {
            existing.remove();
        }
        
        const modalHTML = `
            <div class="modal fade" id="chatbot-loading-modal" tabindex="-1" aria-hidden="true" data-bs-backdrop="static" data-bs-keyboard="false">
                <div class="modal-dialog modal-dialog-centered modal-sm">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">${title}</h5>
                        </div>
                        <div class="modal-body text-center py-4">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p class="mt-3 mb-0">${message}</p>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.body.insertAdjacentHTML('beforeend', modalHTML);
        
        const modal = new ChatbotModal('chatbot-loading-modal', {
            backdrop: 'static',
            keyboard: false
        });
        
        modal.show();
        
        return modal;
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = ChatbotModal;
}

window.ChatbotModal = ChatbotModal;