/**
 * @version 1.0.0
 */

(function () {
    if (window.ChatbotPagination) return;

    class ChatbotPaginationComponent {

        /**
         * @param {object} options
         * @param {number} options.itemsPerPage    
         * @param {number} options.scrollOffset     
         * @param {number} options.scrollDuration   
         * @param {string} options.scrollTarget   
         */
        constructor(options = {}) {
            this.options = {
                itemsPerPage: 4,
                scrollOffset: 100,
                scrollDuration: 300,
                scrollTarget: null,
                ...options
            };

            this._instances = {};
            this._injectStyles();
        }

        /**
         * @param {string} containerId 
         */
        _getOrCreateInstance(containerId) {
            if (!this._instances[containerId]) {
                this._instances[containerId] = {
                    currentPage: 1,
                    allItems: [],
                    filteredItems: null,
                    renderFn: null,
                    containerId: containerId
                };
            }
            return this._instances[containerId];
        }

        /**
         * @param {string}   containerId 
         * @param {Array}    items       
         * @param {Function} renderFn  
         * @param {number}   [page=1]    
         */
        init(containerId, items, renderFn, page = 1) {
            const instance = this._getOrCreateInstance(containerId);
            instance.allItems = items || [];
            instance.filteredItems = null;
            instance.renderFn = renderFn;
            instance.currentPage = page;
            this._render(instance);
        }

        /**
         * @param {string}   containerId  
         * @param {Array}    filteredItems 
         */
        filter(containerId, filteredItems) {
            const instance = this._getOrCreateInstance(containerId);
            instance.filteredItems = filteredItems;
            instance.currentPage = 1;
            this._render(instance);
        }

        /**
         * @param {string} containerId
         */
        clearFilter(containerId) {
            const instance = this._getOrCreateInstance(containerId);
            instance.filteredItems = null;
            instance.currentPage = 1;
            this._render(instance);
        }

        /**
         * @param {string} containerId
         * @param {Array}  items
         */
        update(containerId, items) {
            const instance = this._getOrCreateInstance(containerId);

            if (!instance.renderFn) {
                console.warn(`[ChatbotPagination] update() called before init() for "#${containerId}". Skipped.`);
                return;
            }

            instance.allItems = items || [];
            instance.filteredItems = null;
            instance.currentPage = 1;
            this._render(instance);
        }

        /**
         * @param {string} containerId
         * @param {Array}  newItems
         */
        append(containerId, newItems) {
            const instance = this._getOrCreateInstance(containerId);
            instance.allItems = [...instance.allItems, ...(newItems || [])];
            instance.filteredItems = null;
            this._render(instance);
        }

        /**
         * @param {string}   containerId
         * @param {Function} predicateFn 
         */
        remove(containerId, predicateFn) {
            const instance = this._getOrCreateInstance(containerId);

            if (!instance.renderFn) {
                console.warn(`[ChatbotPagination] remove() called before init() for "#${containerId}". Skipped.`);
                return;
            }

            instance.allItems = instance.allItems.filter(item => !predicateFn(item));
            instance.filteredItems = null;

            const totalPages = this._getTotalPages(instance);
            if (instance.currentPage > totalPages && totalPages > 0) {
                instance.currentPage = totalPages;
            }

            this._render(instance);
        }

        /**
         * @param {string} containerId
         * @param {number} page
         */
        goToPage(containerId, page) {
            const instance = this._getOrCreateInstance(containerId);
            const totalPages = this._getTotalPages(instance);
            if (page < 1 || page > totalPages) return;
            instance.currentPage = page;
            this._render(instance);
        }

        /**
         * @param {string} containerId
         * @returns {{ currentPage, totalPages, totalItems, itemsPerPage }}
         */
        getState(containerId) {
            const instance = this._getOrCreateInstance(containerId);
            const items = instance.filteredItems ?? instance.allItems;
            return {
                currentPage: instance.currentPage,
                totalPages: this._getTotalPages(instance),
                totalItems: items.length,
                itemsPerPage: this.options.itemsPerPage
            };
        }

        /**
         * @param {string} containerId
         */
        destroy(containerId) {
            this._removePagination(containerId);
            delete this._instances[containerId];
        }

        /**
         * @param {string}   containerId  
         * @param {Array}    items       
         * @param {Function} renderFn   
         * @param {number}   itemsPerPage 
         */
        initNested(containerId, items, renderFn, itemsPerPage = 4) {
            const instance = this._getOrCreateInstance(containerId);
            instance.allItems = items || [];
            instance.filteredItems = null;
            instance.renderFn = renderFn;
            instance.currentPage = 1;
            instance.itemsPerPage = itemsPerPage;
            instance.isNested = true;
            this._renderNested(instance);
        }

        _getTotalPagesNested(instance) {
            const items = this._getActiveItems(instance);
            const perPage = instance.itemsPerPage || this.options.itemsPerPage;
            return Math.max(1, Math.ceil(items.length / perPage));
        }

        _renderNested(instance) {
            const { containerId, currentPage, renderFn } = instance;
            const $container = jQuery('#' + containerId);
            if (!$container.length) return;

            const items = this._getActiveItems(instance);
            const perPage = instance.itemsPerPage || this.options.itemsPerPage;
            const totalPages = this._getTotalPagesNested(instance);
            const startIndex = (currentPage - 1) * perPage;
            const pageItems = items.slice(startIndex, startIndex + perPage);

            $container.find('.notebooks-grid-paginated').remove();
            $container.find('.nested-pagination').remove();

            const $grid = jQuery('<div class="notebooks-grid notebooks-grid-paginated"></div>');
            pageItems.forEach(item => renderFn(item, $grid));
            $container.prepend($grid);

            if (totalPages > 1) {
                this._renderNestedPagination(instance, containerId, currentPage, totalPages, items.length);
            }
        }

        _renderNestedPagination(instance, containerId, currentPage, totalPages, totalItems) {
            const perPage = instance.itemsPerPage || this.options.itemsPerPage;
            const from = (currentPage - 1) * perPage + 1;
            const to   = Math.min(currentPage * perPage, totalItems);

            const $pag = jQuery(`
                <div class="nested-pagination d-flex align-items-center justify-content-between mt-2 pt-2 border-top">
                    <button class="btn btn-sm btn-outline-secondary nested-prev" ${currentPage === 1 ? 'disabled' : ''}>
                        <i class="bi bi-chevron-left"></i>
                    </button>
                    <small class="text-muted">
                        ${from}–${to} of ${totalItems}
                        <span class="badge bg-secondary ms-1">${currentPage}/${totalPages}</span>
                    </small>
                    <button class="btn btn-sm btn-outline-secondary nested-next" ${currentPage === totalPages ? 'disabled' : ''}>
                        <i class="bi bi-chevron-right"></i>
                    </button>
                </div>
            `);

            jQuery('#' + containerId).append($pag);

            $pag.find('.nested-prev').off('click').on('click', (e) => {
                e.stopPropagation();
                if (instance.currentPage <= 1) return;
                instance.currentPage--;
                this._renderNested(instance);
            });

            $pag.find('.nested-next').off('click').on('click', (e) => {
                e.stopPropagation();
                if (instance.currentPage >= totalPages) return;
                instance.currentPage++;
                this._renderNested(instance);
            });
        }

        _getActiveItems(instance) {
            return instance.filteredItems ?? instance.allItems;
        }

        _getTotalPages(instance) {
            const items = this._getActiveItems(instance);
            return Math.max(1, Math.ceil(items.length / this.options.itemsPerPage));
        }

        _render(instance) {
            const { containerId, currentPage, renderFn } = instance;
            const $container = jQuery('#' + containerId);

            if (!$container.length) {
                console.warn(`[ChatbotPagination] Container #${containerId} not found.`);
                return;
            }

            const items = this._getActiveItems(instance);
            const totalPages = this._getTotalPages(instance);
            const perPage = this.options.itemsPerPage;
            const startIndex = (currentPage - 1) * perPage;
            const endIndex = Math.min(startIndex + perPage, items.length);
            const pageItems = items.slice(startIndex, endIndex);

            $container.empty();
            this._removePagination(containerId);

            if (items.length === 0) {
                this._renderEmptyState($container, instance.filteredItems !== null);
                return;
            }

            pageItems.forEach(item => renderFn(item, $container));

            if (totalPages > 1) {
                this._renderPagination(containerId, currentPage, totalPages, items.length);
            }
        }

        _renderEmptyState($container, isFiltered) {
            const emptyStateHtml = isFiltered ? `
                <div class="chatbot-pagination-empty text-center py-5" style="animation: fadeIn 0.4s ease-out;">
                    <div class="empty-state-wrapper" style="max-width: 450px; margin: 0 auto;">
                        <div class="empty-state-icon mb-4" style="display: inline-flex; align-items: center; justify-content: center; width: 100px; height: 100px; border-radius: 50%; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3); position: relative;">
                            <div style="position: absolute; inset: -5px; border-radius: 50%; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); opacity: 0.3; filter: blur(15px);"></div>
                            <i class="bi bi-search" style="font-size: 2.5rem; color: white; position: relative; z-index: 1;"></i>
                        </div>
                        <h5 class="fw-bold mb-3" style="color: #1a202c; font-size: 1.5rem;">No Results Found</h5>
                        <p class="text-muted mb-0" style="line-height: 1.7; font-size: 0.95rem; color: #718096;">
                            We couldn't find any items matching your search criteria. Try adjusting your filters or using different keywords.
                        </p>
                    </div>
                </div>
            ` : `
                <div class="chatbot-pagination-empty text-center py-5" style="animation: fadeIn 0.4s ease-out;">
                    <div class="empty-state-wrapper" style="max-width: 450px; margin: 0 auto;">
                        <div class="empty-state-icon mb-4" style="display: inline-flex; align-items: center; justify-content: center; width: 100px; height: 100px; border-radius: 50%; background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); box-shadow: 0 10px 25px rgba(245, 87, 108, 0.3); position: relative;">
                            <div style="position: absolute; inset: -5px; border-radius: 50%; background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); opacity: 0.3; filter: blur(15px);"></div>
                            <i class="bi bi-inbox" style="font-size: 2.5rem; color: white; position: relative; z-index: 1;"></i>
                        </div>
                        <h5 class="fw-bold mb-3" style="color: #1a202c; font-size: 1.5rem;">Nothing Here Yet</h5>
                        <p class="text-muted mb-0" style="line-height: 1.7; font-size: 0.95rem; color: #718096;">
                            There are no items to display at the moment. Start by adding some content to get started.
                        </p>
                    </div>
                </div>
            `;

            $container.html(emptyStateHtml);
        }

       _renderPagination(containerId, currentPage, totalPages, totalItems) {
    const paginationId = `chatbot-pagination-${containerId}`;

    const html = `
        <div class="chatbot-pagination-controls d-flex justify-content-center align-items-center mt-4 gap-3"
             id="${paginationId}"
             data-container="${containerId}">

            <button class="btn-pagination-nav ${currentPage === 1 ? 'disabled' : ''}"
                    ${currentPage === 1 ? 'disabled' : ''}>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M11.354 1.646a.5.5 0 0 1 0 .708L5.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0z"/>
                </svg>
            </button>

            <div class="pagination-counter">
                <span class="current-page">${currentPage}</span>
                <span class="divider">/</span>
                <span class="total-pages">${totalPages}</span>
            </div>

            <button class="btn-pagination-nav ${currentPage === totalPages ? 'disabled' : ''}"
                    ${currentPage === totalPages ? 'disabled' : ''}>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                    <path fill-rule="evenodd" d="M4.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L10.293 8 4.646 2.354a.5.5 0 0 1 0-.708z"/>
                </svg>
            </button>
        </div>
    `;

    jQuery('#' + containerId).after(html);
    this._attachPaginationEvents(containerId, paginationId);
}

        _attachPaginationEvents(containerId, paginationId) {
    const $pagination = jQuery('#' + paginationId);
    const instance = this._instances[containerId];

    $pagination.find('.btn-pagination-nav').first().off('click').on('click', () => {
        if (instance.currentPage <= 1) return;
        instance.currentPage--;
        this._render(instance);
        this._scrollToContainer(containerId);
    });

    $pagination.find('.btn-pagination-nav').last().off('click').on('click', () => {
        const totalPages = this._getTotalPages(instance);
        if (instance.currentPage >= totalPages) return;
        instance.currentPage++;
        this._render(instance);
        this._scrollToContainer(containerId);
    });
}

        _removePagination(containerId) {
            jQuery('#chatbot-pagination-' + containerId).remove();
        }

        _scrollToContainer(containerId) {
            const target = this.options.scrollTarget
                ? jQuery(this.options.scrollTarget)
                : jQuery('#' + containerId);

            if (!target.length) return;

            jQuery('html, body').animate(
                { scrollTop: target.offset().top - this.options.scrollOffset },
                this.options.scrollDuration
            );
        }

       _injectStyles() {
    if (document.getElementById('chatbot-pagination-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'chatbot-pagination-styles';
    style.textContent = `
        @keyframes fadeIn {
            from { 
                opacity: 0; 
                transform: translateY(20px); 
            }
            to { 
                opacity: 1; 
                transform: translateY(0); 
            }
        }
        
        .empty-state-icon {
            transition: transform 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        .chatbot-pagination-empty:hover .empty-state-icon {
            transform: scale(1.08) rotate(5deg);
        }
        
        .chatbot-pagination-empty {
            position: relative;
        }
        
        .empty-state-wrapper {
            position: relative;
            z-index: 1;
        }

        /* New Pagination Styles */
        .chatbot-pagination-controls {
            padding: 0;
            background: transparent;
        }

        .btn-pagination-nav {
            width: 36px;
            height: 36px;
            border-radius: 6px;
            border: 1px solid #e2e8f0;
            background: white;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            color: #64748b;
        }

        .btn-pagination-nav:hover:not(.disabled) {
            background: #f8fafc;
            border-color: #cbd5e1;
            color: #1e293b;
        }

        .btn-pagination-nav.disabled {
            opacity: 0.4;
            cursor: not-allowed;
            pointer-events: none;
        }

        .pagination-counter {
            display: flex;
            align-items: center;
            gap: 4px;
            font-size: 14px;
            font-weight: 500;
            color: #64748b;
            min-width: 60px;
            justify-content: center;
        }

        .pagination-counter .current-page {
            color: #1e293b;
            font-weight: 600;
        }

        .pagination-counter .divider {
            color: #cbd5e1;
        }

        .pagination-counter .total-pages {
            color: #94a3b8;
        }

        /* Dark mode support */
        [data-theme="dark"] .btn-pagination-nav {
            background: #1e293b;
            border-color: #334155;
            color: #cbd5e1;
        }

        [data-theme="dark"] .btn-pagination-nav:hover:not(.disabled) {
            background: #334155;
            border-color: #475569;
            color: #f1f5f9;
        }

        [data-theme="dark"] .pagination-counter {
            color: #cbd5e1;
        }

        [data-theme="dark"] .pagination-counter .current-page {
            color: #f1f5f9;
        }
    `;
    document.head.appendChild(style);
}
    }

    window.ChatbotPagination = new ChatbotPaginationComponent();
})();