/**
 * ChatbotSearch - Reusable Search Component
 * @version 1.0.0
 */

(function () {
    if (window.ChatbotSearch) return;

    class ChatbotSearchComponent {
        
        /**
         * @param {object} options
         * @param {number} options.debounceDelay - Delay para se të search (ms)
         * @param {number} options.minChars - Minimum karaktere për search
         * @param {boolean} options.highlightResults - Highlight search terms
         * @param {string} options.noResultsMessage - Message kur nuk ka rezultate
         */
        constructor(options = {}) {
            this.options = {
                debounceDelay: 300,
                minChars: 1,
                highlightResults: true,
                noResultsMessage: 'No results found',
                ...options
            };

            this._instances = {};
            this._timers = {};
            this._injectStyles();
        }

        /**
         * Initialize search për një container
         * @param {string} searchInputId - ID e input field
         * @param {string} itemsSelector - Selector për items që do të search-ohen
         * @param {Function} filterFn - Function që kthen true nëse item matches
         * @param {Function} onSearch - Callback pas search (optional)
         */
        init(searchInputId, itemsSelector, filterFn, onSearch = null) {
            const instance = {
                searchInputId,
                itemsSelector,
                filterFn,
                onSearch,
                lastQuery: ''
            };

            this._instances[searchInputId] = instance;
            this._attachEventListeners(searchInputId);
            this._createSearchIndicator(searchInputId);
        }

        /**
         * Attach event listeners
         */
        _attachEventListeners(searchInputId) {
            const $input = $(`#${searchInputId}`);
            const instance = this._instances[searchInputId];

            $input.off('input keypress')
                .on('input', () => {
                    clearTimeout(this._timers[searchInputId]);
                    this._timers[searchInputId] = setTimeout(
                        () => this._performSearch(searchInputId),
                        this.options.debounceDelay
                    );
                })
                .on('keypress', (e) => {
                    if (e.which === 13) {
                        e.preventDefault();
                        clearTimeout(this._timers[searchInputId]);
                        this._performSearch(searchInputId);
                    }
                });

            // Clear button functionality
            $input.siblings('.chatbot-search-clear').off('click').on('click', () => {
                this.clear(searchInputId);
            });
        }

        /**
         * Perform search
         */
        _performSearch(searchInputId) {
            const instance = this._instances[searchInputId];
            if (!instance) return;

            const $input = $(`#${searchInputId}`);
            const query = $input.val().trim().toLowerCase();
            
            instance.lastQuery = query;

            // Show/hide clear button
            $input.siblings('.chatbot-search-clear').toggle(query.length > 0);

            if (query.length > 0 && query.length < this.options.minChars) {
                this._updateIndicator(searchInputId, 0, 0, `Type at least ${this.options.minChars} characters`);
                return;
            }

            const $items = $(instance.itemsSelector);
            let visible = 0, hidden = 0;

            if (!query) {
                // No query - show all
                $items.show();
                if (this.options.highlightResults) {
                    this._removeHighlights($items);
                }
                visible = $items.length;
            } else {
                // Filter items
                $items.each((i, item) => {
                    const $item = $(item);
                    const matches = instance.filterFn($item, query);
                    
                    $item.toggle(matches);
                    
                    if (matches) {
                        visible++;
                        if (this.options.highlightResults) {
                            this._highlightText($item, query);
                        }
                    } else {
                        hidden++;
                        if (this.options.highlightResults) {
                            this._removeHighlights($item);
                        }
                    }
                });
            }

            this._updateIndicator(searchInputId, visible, hidden);

            // Callback
            if (instance.onSearch) {
                instance.onSearch({
                    query,
                    visible,
                    hidden,
                    total: visible + hidden
                });
            }
        }

        /**
         * Clear search
         */
        clear(searchInputId) {
            const instance = this._instances[searchInputId];
            if (!instance) return;

            const $input = $(`#${searchInputId}`);
            $input.val('');
            $input.siblings('.chatbot-search-clear').hide();

            const $items = $(instance.itemsSelector);
            $items.show();
            
            if (this.options.highlightResults) {
                this._removeHighlights($items);
            }

            this._updateIndicator(searchInputId, $items.length, 0);

            if (instance.onSearch) {
                instance.onSearch({
                    query: '',
                    visible: $items.length,
                    hidden: 0,
                    total: $items.length
                });
            }
        }

        /**
         * Destroy instance
         */
        destroy(searchInputId) {
            const $input = $(`#${searchInputId}`);
            $input.off('input keypress');
            $input.siblings('.chatbot-search-clear').off('click').remove();
            $(`#chatbot-search-indicator-${searchInputId}`).remove();
            
            clearTimeout(this._timers[searchInputId]);
            delete this._timers[searchInputId];
            delete this._instances[searchInputId];
        }

        /**
         * Get search state
         */
        getState(searchInputId) {
            const instance = this._instances[searchInputId];
            if (!instance) return null;

            const $items = $(instance.itemsSelector);
            const visible = $items.filter(':visible').length;
            
            return {
                query: instance.lastQuery,
                visible,
                hidden: $items.length - visible,
                total: $items.length
            };
        }

        /**
         * Create search indicator
         */
        _createSearchIndicator(searchInputId) {
            const indicatorId = `chatbot-search-indicator-${searchInputId}`;
            
            if ($(`#${indicatorId}`).length) return;

            const $input = $(`#${searchInputId}`);
            const $indicator = $(`
                <div id="${indicatorId}" class="chatbot-search-indicator">
                    <span class="indicator-text"></span>
                </div>
            `);

            $input.closest('.chatbot-search-wrapper, .search-wrapper, .input-group').after($indicator);
        }

        /**
         * Update search indicator
         */
        _updateIndicator(searchInputId, visible, hidden, customMessage = null) {
            const $indicator = $(`#chatbot-search-indicator-${searchInputId}`);
            if (!$indicator.length) return;

            const total = visible + hidden;
            let html = '';

            if (customMessage) {
                html = `<span class="text-muted"><i class="bi bi-info-circle me-1"></i>${customMessage}</span>`;
            } else if (visible === 0 && hidden > 0) {
                html = `<span class="text-warning"><i class="bi bi-exclamation-circle me-1"></i>${this.options.noResultsMessage} (${total} total)</span>`;
            } else if (hidden > 0) {
                html = `<span class="text-info"><i class="bi bi-funnel me-1"></i>Showing ${visible} of ${total}</span>`;
            } else if (total > 0) {
                html = `<span class="text-muted"><i class="bi bi-list-ul me-1"></i>${total} items</span>`;
            }

            $indicator.find('.indicator-text').html(html);
            $indicator.toggle(html.length > 0);
        }

        /**
         * Highlight search terms in text
         */
        _highlightText($item, query) {
            if (!query) return;

            const searchableElements = $item.find('[data-searchable], .page-title, .card-title, .text-muted');
            
            searchableElements.each((i, el) => {
                const $el = $(el);
                const text = $el.text();
                
                // Skip if already highlighted
                if ($el.find('.chatbot-search-highlight').length) return;

                const regex = new RegExp(`(${this._escapeRegex(query)})`, 'gi');
                const highlighted = text.replace(regex, '<mark class="chatbot-search-highlight">$1</mark>');
                
                if (highlighted !== text) {
                    $el.html(highlighted);
                }
            });
        }

        /**
         * Remove highlights
         */
        _removeHighlights($items) {
            $items.find('.chatbot-search-highlight').each((i, mark) => {
                const $mark = $(mark);
                $mark.replaceWith($mark.text());
            });
        }

        /**
         * Escape regex special characters
         */
        _escapeRegex(str) {
            return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }

        /**
         * Inject CSS styles
         */
        _injectStyles() {
            if (document.getElementById('chatbot-search-styles')) return;

            const style = document.createElement('style');
            style.id = 'chatbot-search-styles';
            style.textContent = `
                .chatbot-search-wrapper {
                    position: relative;
                }

                .chatbot-search-clear {
                    position: absolute;
                    right: 10px;
                    top: 50%;
                    transform: translateY(-50%);
                    background: none;
                    border: none;
                    color: #94a3b8;
                    cursor: pointer;
                    padding: 4px 8px;
                    font-size: 18px;
                    display: none;
                    transition: color 0.2s ease;
                }

                .chatbot-search-clear:hover {
                    color: #64748b;
                }

                .chatbot-search-indicator {
                    padding: 8px 12px;
                    margin-top: 8px;
                    border-radius: 6px;
                    background: #f8fafc;
                    font-size: 13px;
                    display: none;
                }

                .chatbot-search-indicator.active {
                    display: block;
                }

                .chatbot-search-highlight {
                    background: #fef08a;
                    padding: 2px 4px;
                    border-radius: 3px;
                    font-weight: 500;
                }

                [data-theme="dark"] .chatbot-search-indicator {
                    background: #1e293b;
                }

                [data-theme="dark"] .chatbot-search-highlight {
                    background: #854d0e;
                    color: #fef08a;
                }
            `;
            document.head.appendChild(style);
        }

        /**
         * Utility: Wrap existing input with search functionality
         */
        wrapInput(inputId) {
            const $input = $(`#${inputId}`);
            if (!$input.length) return;

            if (!$input.parent().hasClass('chatbot-search-wrapper')) {
                $input.wrap('<div class="chatbot-search-wrapper position-relative"></div>');
                $input.after(`
                    <button type="button" class="chatbot-search-clear" title="Clear search">
                        <i class="bi bi-x-circle-fill"></i>
                    </button>
                `);
            }
        }
    }

    // Export global instance
    window.ChatbotSearch = new ChatbotSearchComponent();
})();