<?php
if (!defined('ABSPATH')) exit;

require_once dirname(__DIR__) . '/includes/db.php';
$config_path = dirname(__DIR__) . '/config.json';
$config = [];
if (file_exists($config_path)) {
    $config = json_decode(file_get_contents($config_path), true);
}

global $stripe_url, $prod_id, $customer_email, $title, $base_url;

$stripe_url = sanitize_text_field($config['stripe_url'] ?? '');
$prod_id    = sanitize_text_field($config['prod_id'] ?? '');
$base_url   = sanitize_text_field($config['rag_url'] ?? '');
$title      = get_option('siteurl');

function chatbot_validate_subscription_status($subscription_data) {
    if (!is_array($subscription_data) || empty($subscription_data)) {
        return $subscription_data;
    }
    $status = $subscription_data['status'] ?? null;
    $type = $subscription_data['type'] ?? null;
    $end_date = $subscription_data['currentPeriodEnd'] ?? null;
    
    error_log("Subscription from API - Status: $status, Type: $type, End: $end_date");
    return $subscription_data;
}

function chatbot_plugin_on_activate() {
    global $stripe_url, $prod_id, $customer_email, $title, $base_url;
    $guid = get_option('chatbot_plugin_guid');
    if (!$guid) {
        $guid = wp_generate_uuid4();
        update_option('chatbot_plugin_guid', $guid);
        $expiry_date = date('Y-m-d H:i:s', strtotime('+7 days'));
        update_option('chatbot_plugin_expiry_date', $expiry_date);
    }
    
    $customer_email = wp_get_current_user()->user_email;
    chatbot_items_table();
    chatbot_create_conversations();
    chatbot_create_questions();
    chatbot_create_responses();
    $response = chatbot_stripe_add_trial($stripe_url, $guid, $prod_id, $customer_email, $title);
    
    if (!$response) {
        error_log(" Trial response failed.");
        return;
    }

    error_log(" Trial response: " . print_r($response, true));

    $subscription = $response['subscription'] ?? [];

    if (!empty($subscription)) {
        $subscription = chatbot_validate_subscription_status($subscription);
        $response['subscription'] = $subscription;
    }

    update_option('chatbot_trial_response', $subscription);
    $api_guid = $response['subscription']['appGUID'] ?? $guid;
    if ($api_guid !== $guid) {
        update_option('chatbot_plugin_guid', $api_guid);
        error_log("GUID updated: $guid → $api_guid");
    }

    if (!empty($subscription)) {
        update_option('chatbot_trial_start', $subscription['currentPeriodStart'] ?? '');
        update_option('chatbot_trial_end', $subscription['currentPeriodEnd'] ?? '');
        update_option('chatbot_subscription_id', $subscription['id'] ?? '');
        update_option('chatbot_customer_email', $subscription['customerEmail'] ?? '');
        update_option('chatbot_subscription_status', $subscription['status'] ?? '');
        update_option('chatbot_subscription_type', $subscription['type'] ?? '');

        error_log(" Subscription saved - ID: " . ($subscription['id'] ?? 'N/A') . 
                 " | Status: " . ($subscription['status'] ?? 'N/A') .
                 " | Type: " . ($subscription['type'] ?? 'N/A'));
    }

    if (isset($response['newUserData']['user'])) {
        $user = $response['newUserData']['user'];
        error_log("User data is: " . print_r($user, true));
        $indexing = $response['newUserData']['indexing'] ?? [];

        update_option('chatbot_plugin_indexing', $indexing);

        $login_result = chatbot_plugin_api_login($base_url, $user['email'], $user['password']);
        error_log("Log in result isss" .print_r($login_result,true));

        if (isset($login_result['error'])) {
            error_log(" Login failed: " . $login_result['error']);
            return;
        }

        $access_token = $login_result['access_token'] ?? 
                       $login_result['data']['token'] ?? 
                       $login_result['token'] ?? '';
        
        if (empty($access_token)) {
            error_log(" No access token received. Response: " . print_r($login_result, true));
            return;
        }

        update_option('chatbot_plugin_access_token', $access_token);
        update_option('chatbot_plugin_options', [
            'base_url' => $base_url,
            'username' => $user['email'],
            'password' => $user['password'],
        ]);
        
        error_log(" Login successful - Token: " . substr($access_token, 0, 20) . "...");
    }

    $access_token = get_option('chatbot_plugin_access_token', '');
    error_log("Access token is" .$access_token);
    if (!empty($access_token)) {
        $category = get_option('chatbot_plugin_category', []);
        $stored_guid = $category['name'] ?? '';

        if ($api_guid !== $stored_guid || empty($category['id'])) {
            $sitemap_xml = chatbot_get_sitemap_xml();
            error_log("Sitemap rendered xml " .$sitemap_xml);
            // $new_category = chatbot_plugin_api_create_category(
            //     $base_url,
            //     $access_token,
            //     $api_guid,
            //     $response['newUserData']['indexing']['id'] ?? null,
            //     null,
            //     $sitemap_xml
            // );
            $new_category = chatbot_plugin_api_create_category(
            $base_url,
            $access_token,
            $api_guid,       
            $response['newUserData']['indexing']['id'] ?? null,  
            $sitemap_xml,    
            0                
        );
            error_log("Category new rendered is" .print_r($new_category,true));

            if (isset($new_category['category_id'])) {
                update_option('chatbot_plugin_category', [
                    'id' => $new_category['category_id'],
                    'name' => $api_guid,
                ]);
                if (isset($new_category['summary'])) {
                    update_option('chatbot_summary', sanitize_textarea_field($new_category['summary']));
                    error_log("Chatbot summary is" .$new_category['summary']);
                }
                error_log(" Category created: " . print_r($new_category, true));
            } else {
                error_log(" Category creation failed: " . print_r($new_category, true));
            }
           
        }
    }
    
    add_option('chatbot_plugin_redirect_after_activation', true);
    error_log(" Activation complete with GUID: $api_guid");
}

function chatbot_plugin_set_redirect_flag() {
    add_option('chatbot_plugin_redirect_after_activation', true);
}

add_action('admin_init', 'chatbot_plugin_do_activation_redirect');
    function chatbot_plugin_do_activation_redirect() {
        if (!get_option('chatbot_plugin_redirect_after_activation')) {
            return;
        }
        
        delete_option('chatbot_plugin_redirect_after_activation');
        wp_safe_redirect(admin_url('admin.php?page=mind-web'));
        exit;
    }

add_action('admin_init', 'chatbot_periodic_subscription_check', 5);
    function chatbot_periodic_subscription_check() {
        if (!is_admin()) {
            return;
        }
        
        if (get_transient('chatbot_subscription_last_check')) {
            return;
        }
        
        if (get_transient('chatbot_subscription_checking')) {
            return;
        }
        
        set_transient('chatbot_subscription_checking', true, 60);
        
        try {
            $app_guid = get_option('chatbot_plugin_guid', '');
            if (empty($app_guid)) {
                error_log(" Periodic check: No GUID found");
                return;
            }
            
            $stored_subscription = get_option('chatbot_trial_response', []);
            if (empty($stored_subscription)) {
                error_log(" Periodic check: No stored subscription");
                return;
            }
            $stripe_base_url = chatbot_load_config()['stripe_url'] ?? get_option('chatbot_stripe_base_url', '');
            
            if (empty($stripe_base_url)) {
                error_log(" Periodic check: No stripe URL configured");
                return;
            }
            
            $api_response = chatbot_stripe_active_subscription($stripe_base_url, $app_guid);
            
            if (isset($api_response['error'])) {
                error_log(" Periodic check API error: " . $api_response['error']);
                return;
            }
            
            if (empty($api_response)) {
                error_log(" Periodic check: Empty API response");
                return;
            }
            
            $old_status = $stored_subscription['status'] ?? null;
            $new_status = $api_response['status'] ?? null;
            
            $old_type = $stored_subscription['type'] ?? null;
            $new_type = $api_response['type'] ?? null;
            
            $old_end = $stored_subscription['currentPeriodEnd'] ?? null;
            $new_end = $api_response['currentPeriodEnd'] ?? null;
            
            $needs_update = false;
            
            if ($old_status != $new_status) {
                error_log(" Periodic check: Status changed $old_status → $new_status");
                $needs_update = true;
            }
            
            if ($old_type != $new_type) {
                error_log(" Periodic check: Type changed $old_type → $new_type");
                $needs_update = true;
            }
            
            if ($old_end !== $new_end) {
                error_log(" Periodic check: End date changed $old_end → $new_end");
                $needs_update = true;
            }
            
            if ($needs_update) {
                update_option('chatbot_trial_response', $api_response);
                update_option('chatbot_subscription_status', $new_status);
                update_option('chatbot_subscription_type', $new_type);
                update_option('chatbot_trial_end', $new_end);
                
                error_log(" Periodic check: Subscription updated from API");
            } else {
                error_log(" Periodic check: No changes detected");
            }
            error_log(" Current DB state: Status=$new_status, Type=$new_type, End=$new_end");
            
        } catch (Exception $e) {
            error_log(" Periodic check error: " . $e->getMessage());
        } finally {
            delete_transient('chatbot_subscription_checking');

            set_transient('chatbot_subscription_last_check', true, 30 * MINUTE_IN_SECONDS);
        }
    }


