<?php
if (!defined('ABSPATH')) exit;

    add_action('wp_ajax_chatbot_index_all_items', 'chatbot_plugin_index_items');
        function chatbot_plugin_index_items() {
            check_ajax_referer('chatbot_login_nonce', 'nonce');
            global $wpdb;
            $options = get_option('chatbot_plugin_options', []);
            $base_url = $options['base_url'] ?? '';
            $token = get_option('chatbot_plugin_access_token', '');
            $chunk_size = get_option('chatbot_chunk_size');
            $chunk_overlap = get_option('chatbot_chunk_overlap');
            $header_levels = get_option('chatbot_header_levels');

            $category = get_option('chatbot_plugin_category', null);
            if (!$category || empty($category['id'])) {
                wp_send_json_error('Category not set. Please login first.');
                return;
            }
            $category_id = intval($category['id']);
            $category_name = sanitize_text_field($category['name']);

            if (empty($base_url) || empty($token) || !$category_id) {
                wp_send_json_error('Missing required configuration: Base URL, Token, or Category ID');
                return;
            }

            $fresh_token = chatbot_refresh_token();
            if ($fresh_token) {
                error_log("Token refreshed successfully");
                $token = $fresh_token; 
            } else {
                error_log("Failed to refresh token, using existing token");
            }

            $all_posts = get_posts([
                'numberposts' => -1,
                'post_type' => ['post','page'],
                'post_status' => 'publish'
            ]);

            $results = [];

            foreach ($all_posts as $post) {
                $existing = $wpdb->get_row($wpdb->prepare(
                    "SELECT item_id, category_id FROM {$wpdb->prefix}indexed_items WHERE post_id = %d LIMIT 1",
                    $post->ID
                ));

                $author = get_userdata($post->post_author);
                $author_name = $author ? $author->display_name : 'Unknown';

                if ($existing && $existing->category_id == $category_id) {
                    $results[] = [
                        'post_id' => $post->ID,
                        'item_id' => $existing->item_id,
                        'category_id' => $category_id,
                        'status' => 'already_indexed',
                        'url' => get_permalink($post->ID),
                        'title' => $post->post_title,
                        'post_name' => $post->post_name,
                        'author' => $author_name
                    ];
                    continue; 
                }

                $post_categories = wp_get_post_categories($post->ID, ['fields' => 'names']);
                $post_tags = wp_get_post_tags($post->ID, ['fields' => 'names']);

                $rest_url = rest_url('wp/v2/' . $post->post_type . 's/' . $post->ID);
                $response = wp_remote_get($rest_url);
                
                if (is_wp_error($response)) {
                    error_log('Failed to fetch REST API data for post ' . $post->ID);
                    $wp_json = [];
                } else {
                    $wp_json = json_decode(wp_remote_retrieve_body($response), true);
                }

                $meta = [
                    'categories' => $post_categories,
                    'tags' => $post_tags
                ];
                error_log("Chunk size indexing" .$chunk_size);
                error_log("Chunk overlap index".$chunk_overlap);
                error_log("Header levels index".$header_levels);

                $api_response = chatbot_plugin_api_index_items(
                    $base_url,
                    $token,
                    $category_id,
                    $chunk_size,
                    $chunk_overlap,
                    get_permalink($post->ID),
                    $wp_json,
                    $header_levels,
                    $meta
                );

                if (isset($api_response['error'])) {
                    error_log('[INDEXING] API Error for post ' . $post->ID . ': ' . $api_response['error']);
                    $results[] = [
                        'post_id' => $post->ID,
                        'item_id' => null,
                        'category_id' => $category_id,
                        'status' => 'failed',
                        'error' => $api_response['error'],
                        'url' => get_permalink($post->ID),
                        'title' => $post->post_title,
                        'post_name' => $post->post_name,
                        'author' => $author_name
                    ];
                    continue;
                }

                $new_item_id = $api_response['item_id'] ?? null;

                if (empty($new_item_id)) {
                    error_log('[INDEXING] No item_id returned from API for post ' . $post->ID);
                    $results[] = [
                        'post_id' => $post->ID,
                        'item_id' => null,
                        'category_id' => $category_id,
                        'status' => 'failed',
                        'error' => 'No item_id returned from API',
                        'url' => get_permalink($post->ID),
                        'title' => $post->post_title,
                        'post_name' => $post->post_name,
                        'author' => $author_name
                    ];
                    continue;
                }

                if ($existing) {
                    $updated = $wpdb->update(
                        $wpdb->prefix . 'indexed_items',
                        [
                            'item_id' => $new_item_id,
                            'category_id' => $category_id,
                            'created_at' => current_time('mysql')
                        ],
                        ['post_id' => $post->ID],
                        ['%s', '%d', '%s'],
                        ['%d']
                    );

                    if ($updated === false) {
                        error_log('[INDEXING] Database update failed for post ' . $post->ID);
                        $results[] = [
                            'post_id' => $post->ID,
                            'item_id' => $new_item_id,
                            'category_id' => $category_id,
                            'status' => 'failed',
                            'error' => 'Database update failed',
                            'url' => get_permalink($post->ID),
                            'title' => $post->post_title,
                            'post_name' => $post->post_name,
                            'author' => $author_name
                        ];
                        continue;
                    }
                } else {
                    $inserted = $wpdb->insert(
                        $wpdb->prefix . 'indexed_items',
                        [
                            'post_id' => $post->ID,
                            'item_id' => $new_item_id,
                            'category_id' => $category_id,
                            'created_at' => current_time('mysql')
                        ],
                        ['%d', '%s', '%d', '%s']
                    );

                    if ($inserted === false) {
                        error_log('[INDEXING] Database insert failed for post ' . $post->ID);
                        $results[] = [
                            'post_id' => $post->ID,
                            'item_id' => $new_item_id,
                            'category_id' => $category_id,
                            'status' => 'failed',
                            'error' => 'Database insert failed',
                            'url' => get_permalink($post->ID),
                            'title' => $post->post_title,
                            'post_name' => $post->post_name,
                            'author' => $author_name
                        ];
                        continue;
                    }
                }

                $results[] = [
                    'post_id' => $post->ID,
                    'item_id' => $new_item_id,
                    'category_id' => $category_id,
                    'status' => 'success',
                    'url' => get_permalink($post->ID),
                    'title' => $post->post_title,
                    'post_name' => $post->post_name,
                    'author' => $author_name
                ];
            }
            wp_send_json_success($results);
        }

    add_action('wp_ajax_chatbot_delete_indexed_item', 'chatbot_plugin_delete_indexed_item');
        function chatbot_plugin_delete_indexed_item() {

            check_ajax_referer('chatbot_login_nonce', 'nonce');
            global $wpdb;
            $item_id = sanitize_text_field($_POST['item_id'] ?? '');
            $category_id = intval($_POST['category_id'] ?? 0);

            if (!$item_id || !$category_id) {
                error_log("[UnIndex]  VALIDATION FAILED - Missing data");
                wp_send_json_error(['message' => 'Invalid item_id or category_id']);
            }
            
            $count_before = intval($wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}indexed_items"));
            $row_to_delete = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}indexed_items WHERE item_id = %s AND category_id = %d",
                $item_id,
                $category_id
            ));
            
            if($row_to_delete){
                error_log("[UnIndex]  Row found: " . json_encode($row_to_delete));
            } else {
                error_log("[UnIndex] WARNING: No row found with item_id={$item_id} and category_id={$category_id}");
                $other_category = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}indexed_items WHERE item_id = %s",
                    $item_id
                ));
                
                if($other_category){
                    error_log("[UnIndex] ! Item exists but with different category_id: " . $other_category->category_id);
                }
            }

            $options = get_option('chatbot_plugin_options', []);
            $base_url = $options['base_url'] ?? '';
            $token = get_option('chatbot_plugin_access_token', '');

            if (empty($base_url) || empty($token)) {
                wp_send_json_error(['message' => 'Missing Base URL or Access Token']);
            }

            $api_response = chatbot_plugin_api_delete_indexed_items($base_url, $token, $category_id, $item_id);
            if(!$api_response['success'] && $api_response['http_code'] == 401){
                error_log("[UnIndex]  ACCESS TOKEN EXPIRED. REFRESHING...");
                $new_token = chatbot_refresh_token();
                if($new_token){
                    $api_response = chatbot_plugin_api_delete_indexed_items($base_url, $new_token, $category_id, $item_id);
                }
            }

            if (isset($api_response['error'])) {
                error_log("[UnIndex]  API delete failed: " . $api_response['error']);
                wp_send_json_error(['message' => 'API Error: ' . $api_response['error']]);
            }

            $deleted = $wpdb->delete(
                $wpdb->prefix . 'indexed_items',
                ['item_id' => $item_id, 'category_id' => $category_id],
                ['%s', '%d']
            );

            $wpdb->flush();
            $count_after = intval($wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}indexed_items"));

            if ($deleted === false) {
                wp_send_json_error(['message' => 'Database delete failed: ' . $wpdb->last_error]);
            }

            if($deleted === 0){
                wp_send_json_success([
                    'message' => 'Item removed from API but was not in local database',
                    'deleted_rows' => 0,
                    'count_before' => $count_before,
                    'count_after' => $count_after,
                    'api_response' => $api_response
                ]);
            }


            wp_send_json_success([
                'message' => 'Item unindexed successfully',
                'deleted_rows' => $deleted,
                'count_before' => $count_before,
                'count_after' => $count_after,
                'api_response' => $api_response
            ]);
        }

    add_action('wp_ajax_chatbot_get_stats', 'chatbot_plugin_get_stats');
        function chatbot_plugin_get_stats() {

            check_ajax_referer('chatbot_login_nonce', 'nonce');
            global $wpdb;
            $wpdb->flush();
            $total_posts = wp_count_posts('post')->publish;
            $total_pages = wp_count_posts('page')->publish;
            $total = intval($total_posts + $total_pages);

            $indexed_query = "SELECT COUNT(*) FROM {$wpdb->prefix}indexed_items";
            $indexed_count = intval($wpdb->get_var($indexed_query));
            $indexed_post_ids = $wpdb->get_col("SELECT post_id FROM {$wpdb->prefix}indexed_items");
            $to_index = max(0, $total - $indexed_count);
            
            if ($total > 0) {
                $progress = round(($indexed_count / $total) * 100);
                $progress = min(100, max(0, $progress));
            } else {
                $progress = 0;
            }

            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}indexed_items'");
            if (!$table_exists) {
                error_log("ERROR: indexed_items table does NOT exist!");
            }
            
            wp_send_json_success([
                'total' => $total,
                'indexed' => $indexed_count,
                'to_index' => $to_index,
                'progress' => $progress,
                'debug' => [
                    'total_posts' => intval($total_posts),
                    'total_pages' => intval($total_pages),
                    'indexed_post_ids' => $indexed_post_ids,
                    'table_exists' => !empty($table_exists)
                ]
            ]);
        }

    add_action('wp_ajax_chatbot_get_indexed_table', 'chatbot_plugin_get_indexed_table');
        function chatbot_plugin_get_indexed_table() {
            check_ajax_referer('chatbot_login_nonce', 'nonce');
            
            global $wpdb;
            
            $results = $wpdb->get_results("
                SELECT 
                    i.id, i.item_id, i.post_id, i.category_id,
                    p.ID, p.post_title, p.post_name, p.post_status, p.post_author,
                    u.ID as user_id, u.display_name
                FROM {$wpdb->prefix}indexed_items i
                JOIN {$wpdb->prefix}posts p ON i.post_id = p.ID
                JOIN {$wpdb->prefix}users u ON p.post_author = u.ID
                ORDER BY i.id ASC
            ");

            $html = '';
            if($results){
                foreach($results as $result){
                    $html .= '<tr>
                        <td class="text-danger">' . esc_html($result->id) . '</td>
                        <td>' . esc_html($result->item_id) . '</td>
                        <td>' . esc_html($result->post_id) . '</td>
                        <td>' . esc_html($result->category_id) . '</td>
                        <td>' . esc_html($result->display_name) . '</td>
                        <td>' . esc_html($result->post_title) . '</td>
                        <td>' . esc_html($result->post_name) . '</td>
                        <td>' . esc_html($result->post_status) . '</td>
                        <td>
                            <button class="button button-small button-link-delete unindex-item" 
                                data-item-id="' . esc_attr($result->item_id) . '" 
                                data-category-id="' . esc_attr($result->category_id) . '">
                                <span class="dashicons dashicons-trash"></span> UnIndex
                            </button>
                        </td>
                    </tr>';
                }
            }
            wp_send_json_success($html);
        }

    add_action("wp_ajax_get_default_bot_message", "get_default_bot_message");
    add_action("wp_ajax_nopriv_get_default_bot_message", "get_default_bot_message");
        function get_default_bot_message() {
            if (!session_id()) session_start();

            wp_send_json([
                "show_default" => true,
                "message" => "Hi! How can I assist you today?"
            ]);
        }

    add_action('wp_ajax_chatbot_unindex_all_items', function() {
        check_ajax_referer('chatbot_login_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;
        $table = $wpdb->prefix . 'indexed_items';
        $options = get_option('chatbot_plugin_options', []);
        $base_url = $options['base_url'] ?? '';
        $token = get_option('chatbot_plugin_access_token', '');

        $all_items = $wpdb->get_results("SELECT item_id, category_id FROM $table");

        foreach ($all_items as $item) {
            if ($base_url && $token) {
                chatbot_plugin_api_delete_indexed_items($base_url, $token, $item->category_id, $item->item_id);
            }
        }

        $deleted = $wpdb->query("TRUNCATE TABLE $table");

        if ($deleted !== false) {
            wp_send_json_success(['message' => 'All indexed items have been deleted.']);
        } else {
            wp_send_json_error('Failed to delete items.');
        }
    });

    add_action('wp_ajax_chatbot_refresh_access_token', 'chatbot_handle_refresh_token');
        function chatbot_handle_refresh_token() {
            check_ajax_referer('chatbot_login_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Unauthorized');
                return;
            }
            
            error_log('[Refresh Token] Manual refresh triggered by user');
            
            $new_token = chatbot_refresh_token();
            
            if ($new_token) {
                error_log('[Refresh Token]  Token refreshed successfully');
                wp_send_json_success([
                    'message' => 'Token refreshed successfully',
                    'token' => $new_token
                ]);
            } else {
                error_log('[Refresh Token] ✗ Failed to refresh token');
                wp_send_json_error('Failed to refresh token. Please check your credentials.');
            }
        }

    add_action('wp_ajax_chatbot_check_indexed_items', 'chatbot_check_indexed_items');
    add_action('wp_ajax_nopriv_chatbot_check_indexed_items', 'chatbot_check_indexed_items'); 
        function chatbot_check_indexed_items() {
            
            global $wpdb;
            $table_name = $wpdb->prefix . 'indexed_items';

            if($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
                wp_send_json_error([
                    'message' => 'Indexed items table does not exist.'
                ]);
                return;
            }

            $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
            
            error_log("Indexed items count: {$count}");

            wp_send_json_success([
                'has_items' => ($count > 0),
                'count' => (int)$count
            ]);
        }


    add_action('wp_ajax_chatbot_create_attach_llm_confing_to_indexing', 'chatbot_create_attach_llm_config_to_indexing');
    function chatbot_create_attach_llm_config_to_indexing() {
        check_ajax_referer('chatbot_login_nonce', 'nonce');

        $name         = sanitize_text_field($_POST['name'] ?? '');
        $llm_base_url = sanitize_text_field($_POST['llm_base_url'] ?? '');
        $model_name   = sanitize_text_field($_POST['model_name'] ?? '');
        $provider     = sanitize_text_field($_POST['provider'] ?? '');
        $api_key      = sanitize_textarea_field($_POST['api_key'] ?? '');

        error_log('[LLM-CONNECT]   name         = "' . $name . '"');
        error_log('[LLM-CONNECT]   llm_base_url = "' . $llm_base_url . '"');
        error_log('[LLM-CONNECT]   model_name   = "' . $model_name . '"');
        error_log('[LLM-CONNECT]   provider     = "' . $provider . '"');
        error_log('[LLM-CONNECT]   api_key      = "' . (strlen($api_key) > 0 ? substr($api_key, 0, 8) . '****' : '(empty)') . '"');

        $missing = [];
        if (empty($name))         $missing[] = 'name';
        if (empty($llm_base_url)) $missing[] = 'llm_base_url';
        if (empty($model_name))   $missing[] = 'model_name';
        if (empty($provider))     $missing[] = 'provider';
        if (empty($api_key))      $missing[] = 'api_key';

        if (!empty($missing)) {
            error_log('[LLM-CONNECT]  VALIDATION FAILED. Missing fields: ' . implode(', ', $missing));
            wp_send_json_error(['message' => 'All fields are required. Missing: ' . implode(', ', $missing)]);
            return;
        }
        $chatbot_options = get_option('chatbot_plugin_options', []);
        $base_url        = $chatbot_options['base_url'] ?? '';
        $token           = get_option('chatbot_plugin_access_token', '');

        error_log('[LLM-CONNECT] : chatbot_plugin_options = ' . json_encode($chatbot_options));
        error_log('[LLM-CONNECT]   base_url = "' . $base_url . '"');
        error_log('[LLM-CONNECT]   token    = "' . (strlen($token) > 0 ? substr($token, 0, 20) . '****' : '(empty)') . '"');

        $indexing_options = get_option('chatbot_plugin_indexing', []);
        error_log('[LLM-CONNECT] : Raw indexing option type: ' . gettype($indexing_options));
        error_log('[LLM-CONNECT] : Raw indexing option value: ' . print_r($indexing_options, true));

        if (is_string($indexing_options)) {
            $indexing_options = maybe_unserialize($indexing_options);
            error_log('[LLM-CONNECT] : Unserialized indexing options: ' . print_r($indexing_options, true));
        }

        $indexing_id = $indexing_options['id'] ?? null;
        error_log('[LLM-CONNECT]   indexing_id = ' . var_export($indexing_id, true));

        $config_missing = [];
        if (empty($base_url))   $config_missing[] = 'base_url';
        if (empty($token))      $config_missing[] = 'token';
        if (empty($indexing_id)) $config_missing[] = 'indexing_id';

        if (!empty($config_missing)) {
            error_log('[LLM-CONNECT]  FAILED — Missing config: ' . implode(', ', $config_missing));
            wp_send_json_error(['message' => 'Missing configuration: ' . implode(', ', $config_missing) . '. Please ensure you are logged in and indexing is set up.']);
            return;
        }
        error_log('[LLM-CONNECT] : Config validation passed.');
        
        $result = chatbot_plugin_create_attach_llm_to_config(
            $base_url,
            $token,
            $name,
            $llm_base_url,
            $model_name,
            $provider,
            $api_key,
            $indexing_id
        );

        error_log('[LLM-CONNECT] API response: ' . json_encode($result));

        if (isset($result['error'])) {
            error_log('[LLM-CONNECT]: FAILED — API returned error: ' . $result['error']);
            wp_send_json_error(['message' => 'API Error: ' . $result['error']]);
            return;
        }

        if (empty($result)) {
            error_log('[LLM-CONNECT] : FAILED — API returned empty response.');
            wp_send_json_error(['message' => 'API returned an empty response. Check server logs.']);
            return;
        }

        error_log('[LLM-CONNECT]: API call successful.');

        update_option('chatbot_api_llm_name', $name);
        update_option('chatbot_api_base_url_llm', $llm_base_url);
        update_option('chatbot_api_model_name', $model_name);
        update_option('chatbot_api_provider', $provider);
        update_option('chatbot_api_key', $api_key);
            $llm_config_id = $result['indexing']['llm_config_id'] 
                ?? $result['indexing']['llm_config']['id'] 
                ?? null;

            if ($llm_config_id) {
                update_option('chatbot_llm_config_id', $llm_config_id);

                $indexing_options = get_option('chatbot_plugin_indexing', []);
                if (is_string($indexing_options)) {
                    $indexing_options = maybe_unserialize($indexing_options);
                }
                $indexing_options['llm_config_id'] = $llm_config_id;
                update_option('chatbot_plugin_indexing', $indexing_options);

                error_log('[LLM-CONNECT] Step 10: Saved llm_config_id = ' . $llm_config_id);
            } else {
            error_log('[LLM-CONNECT] Step 10: WARNING — No "id" in API response. chatbot_llm_config_id not updated.');
        }

        wp_send_json_success([
            'message' => 'LLM Configuration successfully created and attached!',
            'data'    => $result
        ]);
    }


    add_action('wp_ajax_chatbot_update_llm_config', 'chatbot_update_llm_config_ajax');
    function chatbot_update_llm_config_ajax() {
        check_ajax_referer('chatbot_login_nonce', 'nonce');

        $name         = sanitize_text_field($_POST['name'] ?? '');
        $llm_base_url = sanitize_text_field($_POST['llm_base_url'] ?? '');
        $model_name   = sanitize_text_field($_POST['model_name'] ?? '');
        $provider     = sanitize_text_field($_POST['provider'] ?? '');
        $api_key      = sanitize_textarea_field($_POST['api_key'] ?? '');

        $missing = [];
        if (empty($name))         $missing[] = 'name';
        if (empty($llm_base_url)) $missing[] = 'llm_base_url';
        if (empty($model_name))   $missing[] = 'model_name';
        if (empty($provider))     $missing[] = 'provider';
        if (empty($api_key))      $missing[] = 'api_key';

        if (!empty($missing)) {
            wp_send_json_error(['message' => 'All fields are required. Missing: ' . implode(', ', $missing)]);
            return;
        }

        $chatbot_options  = get_option('chatbot_plugin_options', []);
        $base_url         = $chatbot_options['base_url'] ?? '';
        $token            = get_option('chatbot_plugin_access_token', '');
        $llm_config_id    = get_option('chatbot_llm_config_id');
        $indexing_options = get_option('chatbot_plugin_indexing', []);
        $user_id          = $indexing_options['user_id'] ?? null;

        if (empty($base_url) || empty($token) || empty($llm_config_id) || empty($user_id)) {
            wp_send_json_error(['message' => 'Missing configuration or LLM not created yet.']);
            return;
        }

        $result = chatbot_plugin_update_llm_config(
            $base_url,
            $token,
            $llm_config_id,
            $name,
            $api_key,
            $llm_base_url,
            $model_name,
            $user_id
        );

        if (isset($result['error'])) {
            wp_send_json_error(['message' => 'API Error: ' . $result['error']]);
            return;
        }

        if (empty($result)) {
            wp_send_json_error(['message' => 'API returned empty response']);
            return;
        }

        update_option('chatbot_api_llm_name', $name);
        update_option('chatbot_api_base_url_llm', $llm_base_url);
        update_option('chatbot_api_model_name', $model_name);
        update_option('chatbot_api_provider', $provider);
        update_option('chatbot_api_key', $api_key);

        $indexing_options['llm_config_id'] = $llm_config_id;
        update_option('chatbot_plugin_indexing', $indexing_options);

        update_option('chatbot_llm_config_id', $llm_config_id);

        wp_send_json_success([
            'message' => 'LLM configuration successfully updated!',
            'data'    => $result
        ]);
    }

add_action('wp_ajax_chatbot_delete_llm_config_ajax', 'chatbot_delete_llm_config_ajax');
function chatbot_delete_llm_config_ajax() {
    check_ajax_referer('chatbot_login_nonce', 'nonce');
    
    $chatbot_options = get_option('chatbot_plugin_options', []);
    $base_url = $chatbot_options['base_url'] ?? '';
    $token = get_option('chatbot_plugin_access_token', '');
    $llm_config_id = get_option('chatbot_llm_config_id');
    
    error_log('[LLM-DELETE-AJAX] Starting deletion process');
    error_log('[LLM-DELETE-AJAX] LLM Config ID: ' . $llm_config_id);
    
    if (empty($base_url) || empty($token) || empty($llm_config_id)) {
        error_log('[LLM-DELETE-AJAX] Missing required data');
        wp_send_json_error(['message' => 'Missing configuration or LLM config ID.']);
        return;
    }
    
    $result = chatbot_plugin_delete_llm_config(
        $base_url,
        $token,
        $llm_config_id
    );
    
    error_log('[LLM-DELETE-AJAX] API Result: ' . print_r($result, true));
    
    if (isset($result['error'])) {
        error_log('[LLM-DELETE-AJAX] Error occurred: ' . $result['error']);
        wp_send_json_error(['message' => 'API Error: ' . $result['error']]);
        return;
    }
    
    delete_option('chatbot_api_llm_name');
    delete_option('chatbot_api_base_url_llm');
    delete_option('chatbot_api_model_name');
    delete_option('chatbot_api_provider');
    delete_option('chatbot_api_key');
    delete_option('chatbot_llm_config_id');
    
    error_log('[LLM-DELETE-AJAX] Options deleted successfully');
    
    wp_send_json_success(['message' => 'LLM configuration disconnected successfully!']);
}

function enqueue_chatbot_llm_script() {
        wp_enqueue_script(
        'chatbot-create-llm',
        plugin_dir_url(__DIR__) . 'assets/js/chatbot_create_attach_llm_config.js',
        ['jquery', 'bootstrap-js'], 
        '1.0',
        true
    );

    wp_localize_script('chatbot-create-llm', 'chatbotLLMConfig', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('chatbot_login_nonce'),
        'llm_config_id' =>  get_option('chatbot_llm_config_id'),
    ]);
}
add_action('admin_enqueue_scripts', 'enqueue_chatbot_llm_script');

