<?php
if (!defined('ABSPATH')) exit;

function chatbot_plugin_api_login($base_url, $username, $password) {
    $url = rtrim($base_url, '/') . '/auth/login';
    $payload = [
        'username' => $username,
        'password' => $password
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload)); 
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/x-www-form-urlencoded',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    $decoded = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'error' => 'Invalid JSON: ' . json_last_error_msg(),
            'raw' => $response_body,
            'http_status' => $http_status
        ];
    }

    $decoded['http_status'] = $http_status;
    return $decoded;
}


function chatbot_plugin_api_create_category(
    $base_url,
    $token,
    $name,
    $indexing_id,
    $sitemap_xml,
    $parent_id = 0
) {
    $url = rtrim($base_url, '/') . '/categories/wordpress';

    $query_params = [
        'name' => $name,
        'indexing_id' => intval($indexing_id),
        'parent_id' => intval($parent_id),
    ];

    $url .= '?' . http_build_query($query_params);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);

    $body = $sitemap_xml ?: '<sitemap></sitemap>';
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);

    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $token,
        'Content-Type: application/xml',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); 
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $response_body = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($curl_error) {
        error_log('Category creation cURL error: ' . $curl_error);
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    if ($http_code >= 400) {
        error_log('Category creation HTTP error ' . $http_code . ': ' . $response_body);
        return ['error' => 'HTTP Error ' . $http_code, 'response' => $response_body];
    }

    $result = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log('Invalid JSON response: ' . $response_body);
        return ['error' => 'Invalid response format'];
    }

    error_log('Category created successfully: ' . print_r($result, true));
    return $result;
}

function chatbot_plugin_api_get_categories($base_url, $token) {
    $url = rtrim($base_url, '/') . '/categories/';
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $token,
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    return json_decode($response_body, true);
}

function chatbot_plugin_api_index_items($base_url, $token, $category_id, $chunk_size, $chunk_overlap, $url, $wp_json, $header_levels, $meta) {
    $api_url = rtrim($base_url, '/') . '/api/indexingitems/wordpress';
    $payload = [
        'category_id' => $category_id,
        'chunk_size' => $chunk_size,
        'chunk_overlap' => $chunk_overlap,
        'url' => $url,
        'wp_json' => json_encode($wp_json, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
        'header_levels' => $header_levels,
        'meta' => json_encode($meta, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    ];
    
    error_log("[API] Indexing URL: " . $url);
    error_log("[API] Category ID: " . $category_id);
    error_log("[API] WP JSON size: " . strlen($payload['wp_json']) . " bytes");
    error_log("[API] Meta: " . $payload['meta']);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $token,
        'Accept: application/json',
        'Content-Type: application/x-www-form-urlencoded'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 0);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0); 

    $response_body = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($curl_error) {
        error_log('[API] cURL Error: ' . $curl_error);
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    error_log("[API] HTTP Code: " . $http_code);
    error_log("[API] Response: " . substr($response_body, 0, 500)); 

    if ($http_code !== 200 && $http_code !== 201) {
        error_log('[API] Error - HTTP ' . $http_code . ': ' . $response_body);
        return [
            'error' => 'HTTP ' . $http_code,
            'response' => $response_body,
            'http_code' => $http_code
        ];
    }

    $decoded = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log('[API] JSON decode error: ' . json_last_error_msg());
        error_log('[API] Response body: ' . $response_body);
        return ['error' => 'Invalid JSON response: ' . json_last_error_msg()];
    }

    if (!isset($decoded['item_id']) || empty($decoded['item_id'])) {
        error_log('[API] Warning: No item_id in response');
        error_log('[API] Response: ' . print_r($decoded, true));
        return ['error' => 'No item_id in API response'];
    }

    error_log('[API] Success - Item ID: ' . $decoded['item_id']);
    return $decoded;
}

function chatbot_plugin_api_delete_indexed_items($base_url, $token, $category_id, $item_id) {
    $api_url = rtrim($base_url, '/') . "/api/{$category_id}/{$item_id}";

    $response = wp_remote_request($api_url, [
        'method'    => 'DELETE',
        'headers'   => [
            'Authorization' => 'Bearer ' . $token,
            'Content-Type'  => 'application/json',
        ],
        'timeout'   => 20,
    ]);

    if (is_wp_error($response)) {
        return ['error' => $response->get_error_message()];
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    
    if ($code >= 400) {
        return ['error' => "HTTP {$code}: {$body}"];
    }

    return json_decode($body, true) ?: ['success' => true];
}

function chatbot_refresh_token() {
    $options = get_option('chatbot_plugin_options', []);

    $base_url = $options['base_url'] ?? '';
    $username = $options['username'] ?? '';
    $password = $options['password'] ?? '';
    error_log("Plugin option for refresh token " .$base_url);
    error_log("Plugin option for refresh token username" .$username);
    error_log("Plugin option for password is" .$password);
 
    if (!$base_url || !$username || !$password) {
        error_log("Missing login credentials for token refresh.");
        return false;
    }

    $login = chatbot_plugin_api_login($base_url, $username, $password);

    if (!isset($login['access_token'])) {
        error_log("Token refresh failed.");
        return false;
    }

    update_option('chatbot_plugin_access_token', $login['access_token']);
    error_log("Token refreshed successfully.");

    return $login['access_token'];
}

function chatbot_plugin_knowledge_repository_get_token() {
    $options = get_option('chatbot_knowledge_repository_options', []);

    $token = $options['token'] ?? '';
    $expiry = $options['token_expiry'] ?? 0;

    if (!$token || time() >= $expiry) {
        $token = chatbot_plugin_knowledge_repository_refresh_token();
    }

    return $token;
}

function chatbot_plugin_knowledge_repository_refresh_token() {
    $options = get_option('chatbot_knowledge_repository_options', []);
    $base_url = $options['base_url'] ?? '';
    $username = $options['username'] ?? '';
    $password = $options['password'] ?? '';
    
    if (!$base_url || !$username || !$password) {
        error_log("Missing credentials for token refresh.");
        error_log("Base URL: " . ($base_url ?: 'MISSING'));
        error_log("Username: " . ($username ?: 'MISSING'));
        error_log("Password: " . ($password ? 'EXISTS' : 'MISSING'));
        return false;
    }
    
    error_log("Attempting to refresh token for user: " . $username);
    error_log("   Base URL: " . $base_url);
    
    $login = chatbot_plugin_knowledge_repository_login_with_cookie($base_url, $username, $password);
    
    error_log("Login response during refresh: " . print_r($login, true));
    
    if (isset($login['error'])) {
        error_log(" Token refresh failed with error: " . $login['error']);
        return false;
    }
    
    if (!isset($login['data']['token'])) {
        error_log(" Token refresh failed: No token in response");
        error_log("   Response data: " . print_r($login, true));
        return false;
    }
    
    $new_token = $login['data']['token'];
    error_log(" New token obtained: " . substr($new_token, 0, 30) . "...");
    error_log("   Token length: " . strlen($new_token) . " chars");
    
    $options['token'] = $new_token;
    $updated = update_option('chatbot_knowledge_repository_options', $options);
    
    if ($updated) {
        error_log("Token saved successfully in database.");
    } else {
        error_log(" update_option returned false (token might be same as before)");
    }
    
    $verify = get_option('chatbot_knowledge_repository_options', []);
    $saved_token = $verify['token'] ?? '';
    
    if ($saved_token === $new_token) {
        error_log(" Verified: Token correctly saved in database");
    } else {
        error_log(" WARNING: Saved token doesn't match new token!");
        error_log("   Expected: " . substr($new_token, 0, 30) . "...");
        error_log("   Got: " . substr($saved_token, 0, 30) . "...");
    }
    
    return $new_token;
}

function chatbot_get_api_response_stream($base_url, $token, $question, $category_id, $category_ids, $top_k, $app_guid, $callback) {
    $ch = curl_init(rtrim($base_url, '/') . '/ask/stream');

    $postData = [
        'question' => $question,
        'category_id' => $category_id,
        'top_k' => $top_k,
        'category_ids' => $category_ids,
        'distance' => 1
    ];

    $headers = [
        'Authorization: Bearer ' . $token,
        'Content-Type: application/json',
        'Accept: text/event-stream'
    ];
    if (!empty($app_guid)) $headers[] = 'X-Subscription-Guid: ' . $app_guid;

    curl_setopt_array($ch, [
        CURLOPT_POST => 1,
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_WRITEFUNCTION => $callback,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_BUFFERSIZE => 128,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
    ]);

    $result = curl_exec($ch);
    $curl_error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return [
        'success' => ($http_code === 200 && !$curl_error),
        'http_code' => $http_code,
        'error' => $curl_error
    ];
}

function chatbot_ask_page_based($base_url, $token, $question, $category_id, $category_ids, $top_k, $app_guid, $callback,$items_ids){
    $ch = curl_init(rtrim($base_url, '/') . '/ask/stream_pro');

    $postData = [
        'question' => $question,
        'category_id' => $category_id,
        'top_k' => $top_k,
        'category_ids' => $category_ids,
        'distance' => 1,
        'items_id' => $items_ids
        ];

    $headers = [
        'Authorization: Bearer ' . $token,
        'Content-Type: application/json',
        'Accept: text/event-stream'
    ];
    if (!empty($app_guid)) $headers[] = 'X-Subscription-Guid: ' . $app_guid;

    curl_setopt_array($ch, [
        CURLOPT_POST => 1,
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_WRITEFUNCTION => $callback,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_BUFFERSIZE => 128,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
    ]);

    $result = curl_exec($ch);
    $curl_error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return [
        'success' => ($http_code === 200 && !$curl_error),
        'http_code' => $http_code,
        'error' => $curl_error
    ];

}

function chatbot_stripe_add_trial($stripe_base_url, $app_guid,$product_id,$customer_email,$title) {
    $url = rtrim($stripe_base_url, '/') . '/Subscription/AddTrialWithUser';
    error_log("stripe url is" .$url);
    $payload = ['AppGuid' => $app_guid , 'StripeProductId' => $product_id, 'CustomerEmail' => $customer_email, 'title'=>$title];
    $json_payload = json_encode($payload);

    error_log("Stripe URL: $url");
    error_log("Payload: $json_payload");

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json_payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    error_log("HTTP status: $http_status");
    error_log("Response body: $response_body");

    if ($curl_error) {
        error_log("cURL Error: $curl_error");
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    $decoded = json_decode($response_body, true);
    if(json_last_error() !== JSON_ERROR_NONE){
        return ['error' => 'Invalid JSON response: ' . json_last_error_msg(), 'raw' => $response_body];
    }

    return $decoded;
}

function chatbot_stripe_add_trial_withoutUser($stripe_base_url, $app_guid,$product_id,$customer_email) {
    $url = rtrim($stripe_base_url, '/') . '/Subscription/AddTrial';
    error_log("stripe url is" .$url);
    $payload = ['AppGuid' => $app_guid , 'StripeProductId' => $product_id, 'CustomerEmail' => $customer_email];
    $json_payload = json_encode($payload);

    error_log("Stripe URL: $url");
    error_log("Payload: $json_payload");

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json_payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    error_log("HTTP status: $http_status");
    error_log("Response body: $response_body");

    if ($curl_error) {
        error_log("cURL Error: $curl_error");
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    $decoded = json_decode($response_body, true);
    if(json_last_error() !== JSON_ERROR_NONE){
        return ['error' => 'Invalid JSON response: ' . json_last_error_msg(), 'raw' => $response_body];
    }

    return $decoded;
}

function chatbot_stripe_active_subscription($stripe_base_url, $app_guid) {

  $url = rtrim($stripe_base_url, '/') . '/Subscription/GetActiveSubscription/' . $app_guid;
  error_log("url subscription active" .$url);
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }
    return json_decode($response_body, true);
}

function chatbot_get_all_subscriptions($stripe_base_url,$app_guid){
    $url = rtrim($stripe_base_url, '/') . '/Subscription/' . $app_guid;
    error_log("url of chatbot get all subscriptions" .$url);
    
    $ch= curl_init();
    curl_setopt($ch,CURLOPT_URL,$url);
      curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }
    return json_decode($response_body, true);
}

function chatbot_get_history_subscription($stripe_base_url,$app_guid){
    $url = rtrim($stripe_base_url, '/') .'/Subscription/GetHistory/' . $app_guid;
    error_log("url of chatbot get all subscriptions" .$url);
    
    $ch= curl_init();
    curl_setopt($ch,CURLOPT_URL,$url);
      curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }
    return json_decode($response_body, true);
    
}

function chatbot_plugin_cancel_subscription($stripe_base_url, $app_guid,$subscription_id, $cancel="true"){
    $url = rtrim($stripe_base_url, '/') . '/Subscription/CancelSubscription/' . $subscription_id ;
    error_log("url for cancel isss" .$url);
    $payload = ['AppGuid' => $app_guid, 'Cancel' => $cancel];
    error_log("psyload is" . json_encode(@$payload));

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }
    return json_decode($response_body, true);
}

function chatbot_plugin_reactivate_subscription($stripe_base_url, $app_guid,$subscription_id, $cancel="false"){
    $url = rtrim($stripe_base_url, '/') . '/Subscription/CancelSubscription/' . $subscription_id;
    $payload = ['AppGuid' => $app_guid, 'Cancel' => $cancel];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $response_body = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);
    if ($curl_error) {
        return ['error' => 'cURL Error: ' . $curl_error];
    }
    return json_decode($response_body, true);
}

function chatbot_plugin_knowledge_repository_login_with_cookie($knowledge_repository_base_url, $username, $password) {
    $login_url = rtrim($knowledge_repository_base_url, '/') . '/api/Identity/LoginWithCredentials';
    $payload = json_encode(['userName' => $username, 'password' => $password]);
    $ch = curl_init($login_url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json','Accept: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);


    if ($curl_error) {
        error_log("cURL Error during login: " . $curl_error);
        return ['error' => 'Connection error: ' . $curl_error];
    }
    error_log("http code during login: " . $http_code);

    if ($http_code !== 200) {
        error_log("Login failed with HTTP code: " . $http_code . " Response: " . $response);
        return ['error' => 'Login failed with status code: ' . $http_code];
    }
    error_log("Login response: " . $response);
    error_log("Login HTTP code: " . $http_code);

    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Invalid JSON response: " . json_last_error_msg());
        return ['error' => 'Invalid JSON response from login'];
    }

    if (!isset($data['token'])) {
        error_log("No token in login response: " . print_r($data, true));
        return ['error' => 'No authentication token received'];
    }

    return ['data' => $data];
}

function chatbot_plugin_get_notebooks($knowledge_repository_base_url, $token) {
    $url = rtrim($knowledge_repository_base_url, '/') . '/api/ApiNotebook';
    $token = trim($token); 

    error_log("URL: " . $url);
    error_log("Token prefix: " . substr($token, 0, 20)); 

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $token,
        'Accept: text/plain',
    ]);
    
    $response   = curl_exec($ch);
    $http_code  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    error_log("HTTP CODE notebooks: " . $http_code);

    if ($curl_error) {
        error_log("cURL Error fetching notebooks: " . $curl_error);
        return ['error' => 'Connection error: ' . $curl_error];
    }

    if (strpos($response, '<!DOCTYPE html') !== false) {
        error_log("Got HTML instead of JSON. Response: " . substr($response, 0, 500));
        return ['error' => 'Server returned HTML instead of JSON (check URL or auth).'];
    }

    if ($http_code === 401) {
        error_log("401 Unauthorized - token expired or invalid");
        return [
            'status' => 401,
            'error' => 'Unauthorized - token expired'
        ];
    }

    if ($http_code !== 200) {
        error_log("Get notebooks failed with HTTP code: " . $http_code . " Response: " . substr($response, 0, 500));
        return [
            'status' => $http_code,
            'error' => 'Failed to fetch notebooks. Status code: ' . $http_code
        ];
    }

    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Invalid JSON response from notebooks: " . json_last_error_msg() . " Raw: " . substr($response, 0, 500));
        return ['error' => 'Invalid JSON response from notebooks'];
    }

    $notebooks = $data['notebooks'] ?? $data;
    
    if (is_array($notebooks) && isset($notebooks['notebooks'])) {
        $notebooks = $notebooks['notebooks'];
    }

    return ['notebooks' => $notebooks];
}

function chatbot_plugin_create_attach_llm_to_config($base_url, $token, $name, $llm_base_url, $model_name, $provider, $api_key, $indexing_id) {

    $url = rtrim($base_url, '/') . '/indexing/create-and-attach-llm-config-to-indexing';
    error_log('[LLM-CURL] Target URL: ' . $url);

    $payload = [
        'name'          => $name,
        'llm_base_url'  => $llm_base_url,
        'model_name'    => $model_name,
        'provider'      => $provider,
        'api_key'       => $api_key,
        'indexing_id'   => $indexing_id
    ];

    error_log('[LLM-CURL] Payload (api_key masked): ' . json_encode(array_merge($payload, ['api_key' => substr($api_key, 0, 8) . '****'])));

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $token,
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_VERBOSE, true);

    $curl_verbose = fopen('php://temp', 'rw');
    curl_setopt($ch, CURLOPT_STDERR, $curl_verbose);

    $response_body = curl_exec($ch);

    $http_code   = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error  = curl_error($ch);
    $curl_errno  = curl_errno($ch);
    $connect_time = curl_getinfo($ch, CURLINFO_CONNECT_TIME);
    $total_time   = curl_getinfo($ch, CURLINFO_TOTAL_TIME);

    rewind($curl_verbose);
    $curl_verbose_log = stream_get_contents($curl_verbose);
    fclose($curl_verbose);

    curl_close($ch);

    if ($curl_error) {
        error_log('[LLM-CURL] FAILED — cURL error: ' . $curl_error);
        return ['error' => 'cURL Error: ' . $curl_error];
    }

    if ($http_code < 200 || $http_code >= 300) {
        return ['error' => 'API returned HTTP ' . $http_code . '. Body: ' . $response_body];
    }

    $decoded = json_decode($response_body, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log('[LLM-CURL] FAILED — JSON decode error: ' . json_last_error_msg());
        return ['error' => 'Failed to parse API response: ' . json_last_error_msg()];
    }

    error_log('[LLM-CURL] Decoded response: ' . json_encode($decoded));
    return $decoded;
}

function chatbot_plugin_update_llm_config($base_url, $token, $llm_id, $name, $api_key, $llm_base_url, $model_name, $user_id) {

    $url = rtrim($base_url, '/') . "/llm-configs/{$llm_id}";
    error_log('[LLM-CURL] Target URL: ' . $url);
    $payload =[
        'name' => $name,
        'api_key' => $api_key,
        'llm_base_url' => $llm_base_url,
        'model_name' => $model_name,
        'user_id' => $user_id
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT"); 
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer {$token}",
        "Content-Type: application/json"
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    if (curl_errno($ch)) {
        error_log('[LLM-CURL] Error: ' . curl_error($ch));
    } else {
        error_log('[LLM-CURL] Response Code: ' . $http_code);
        error_log('[LLM-CURL] Response Body: ' . $response);
    }

    curl_close($ch);

    return json_decode($response, true);
}

function chatbot_plugin_delete_llm_config($base_url, $token, $llm_id) {
    $url = rtrim($base_url, '/') . "/llm-configs/{$llm_id}";
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "DELETE");
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer {$token}",
        "Content-Type: application/json"
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        error_log('[LLM-CURL] Error: ' . $error);
        curl_close($ch);
        return ['error' => $error];
    }
    
    curl_close($ch);
    
    error_log('[LLM-CURL] Response Code: ' . $http_code);
    error_log('[LLM-CURL] Response Body: ' . $response);
    
    if ($http_code >= 200 && $http_code < 300) {
        if (empty($response)) {
            return ['success' => true, 'message' => 'LLM config deleted'];
        }
        
        $decoded = json_decode($response, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }
        
        return ['success' => true, 'message' => 'LLM config deleted'];
    }
    
    return ['error' => 'HTTP Error: ' . $http_code . ' - ' . $response];
}


function chatbot_downloadInvoice($stripe_base_url, $invoiceNumber){
    $url = rtrim($stripe_base_url, '/') . '/Invoices/GetInvoicePdfUrl/' . urlencode($invoiceNumber);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
    ]);

    $response = curl_exec($ch);

    if (curl_errno($ch)) {
        curl_close($ch);
        return ['error' => 'Request error: ' . curl_error($ch)];
    }

    curl_close($ch);

    $response = trim($response);

    if (!empty($response)) {
        return ['url' => $response];
    }

    return ['error' => 'No URL returned from API'];
}


