<?php
/**
 * Reusable Modal Component
 * 
 * @package Chatbot_Plugin
 * @version 1.0.0
 */

if (!defined('ABSPATH')) exit;

class Chatbot_Modal {
    
    /**
     * Modal ID
     * @var string
     */
    private $id;
    
    /**
     * Modal title
     * @var string
     */
    private $title;
    
    /**
     * Modal size (sm, md, lg, xl)
     * @var string
     */
    private $size;
    
    /**
     * Constructor
     * 
     * @param string $id Modal ID
     * @param string $title Modal title
     * @param string $size Modal size (sm, md, lg, xl)
     */
    public function __construct($id, $title, $size = 'md') {
        $this->id = $id;
        $this->title = $title;
        $this->size = $size;
    }
    
    /**
     * Render complete modal
     * 
     * @param string $content Modal body content
     * @param string $footer Modal footer content (optional)
     * @param array $attributes Additional modal attributes
     */
    public function render($content, $footer = '', $attributes = []) {
        $default_attributes = [
            'data-bs-backdrop' => 'static',
            'data-bs-keyboard' => 'true'
        ];
        
        $attributes = array_merge($default_attributes, $attributes);
        $attr_string = '';
        
        foreach ($attributes as $key => $value) {
            $attr_string .= sprintf(' %s="%s"', esc_attr($key), esc_attr($value));
        }
        ?>
        <div class="modal fade" 
             id="<?php echo esc_attr($this->id); ?>" 
             tabindex="-1" 
             aria-hidden="true"
             <?php echo $attr_string; ?>>
            <div class="modal-dialog modal-dialog-centered modal-<?php echo esc_attr($this->size); ?>">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title"><?php echo esc_html($this->title); ?></h5>
                        <button type="button" 
                                class="btn-close" 
                                data-bs-dismiss="modal" 
                                aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <?php echo $content; ?>
                    </div>
                    <?php if ($footer): ?>
                    <div class="modal-footer">
                        <?php echo $footer; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Create a confirmation modal (static method)
     * 
     * @param string $id Modal ID
     * @param string $title Modal title
     * @param string $message Confirmation message
     * @param string $confirm_text Confirm button text
     * @param string $confirm_class Confirm button CSS class
     */
    public static function confirm($id, $title, $message, $confirm_text = 'Confirm', $confirm_class = 'btn-danger') {
        $content = '<p>' . esc_html($message) . '</p>';
        
        $footer = sprintf(
            '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
             <button type="button" class="btn %s" id="%s-confirm" data-confirm>%s</button>',
            esc_attr($confirm_class),
            esc_attr($id),
            esc_html($confirm_text)
        );
        
        $modal = new self($id, $title, 'sm');
        $modal->render($content, $footer);
    }
    
    /**
     * Create a loading modal (static method)
     * 
     * @param string $id Modal ID
     * @param string $title Modal title
     * @param string $message Loading message
     */
    public static function loading($id, $title, $message = 'Loading...') {
        $content = '
            <div class="text-center py-4">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3 mb-0">' . esc_html($message) . '</p>
            </div>
        ';
        
        $modal = new self($id, $title);
        $modal->render($content, '', ['data-bs-backdrop' => 'static', 'data-bs-keyboard' => 'false']);
    }
    
    /**
     * Create a success modal (static method)
     * 
     * @param string $id Modal ID
     * @param string $title Modal title
     * @param string $message Success message
     */
    public static function success($id, $title, $message) {
        $content = '
            <div class="text-center py-4">
                <div class="mb-3">
                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#10b981" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <polyline points="8 12 11 15 16 9"></polyline>
                    </svg>
                </div>
                <p class="mb-0">' . esc_html($message) . '</p>
            </div>
        ';
        
        $footer = '<button type="button" class="btn btn-primary" data-bs-dismiss="modal">OK</button>';
        
        $modal = new self($id, $title, 'sm');
        $modal->render($content, $footer);
    }
    
    /**
     * Create an error modal (static method)
     * 
     * @param string $id Modal ID
     * @param string $title Modal title
     * @param string $message Error message
     */
    public static function error($id, $title, $message) {
        $content = '
            <div class="text-center py-4">
                <div class="mb-3">
                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#dc2626" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <line x1="15" y1="9" x2="9" y2="15"></line>
                        <line x1="9" y1="9" x2="15" y2="15"></line>
                    </svg>
                </div>
                <p class="mb-0">' . esc_html($message) . '</p>
            </div>
        ';
        
        $footer = '<button type="button" class="btn btn-danger" data-bs-dismiss="modal">Close</button>';
        
        $modal = new self($id, $title, 'sm');
        $modal->render($content, $footer);
    }
}