<?php
if ( ! defined( 'ABSPATH' ) ) exit;

$config_path = dirname( __DIR__ ) . '/config.json';
$config      = [];
if ( file_exists( $config_path ) ) {
    $config = json_decode( file_get_contents( $config_path ), true );
}

/* ═══════════════════════════════════════════════════════════════
   LOGIN
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_knowledge_repository_login', 'chatbot_knowledge_repository_login' );
function chatbot_knowledge_repository_login() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    global $config;
    $base_url = sanitize_text_field( $_POST['base_url'] ?? ( $config['notebooks_url'] ?? 'https://spaces.mind.smartprocesses.cloud' ) );
    $username = sanitize_text_field( $_POST['username'] ?? '' );
    $password = sanitize_text_field( $_POST['password'] ?? '' );

    if ( ! $base_url || ! $username || ! $password ) {
        wp_send_json_error( [ 'message' => 'Please fill in all fields.' ] );
    }
    if ( ! filter_var( $base_url, FILTER_VALIDATE_URL ) ) {
        wp_send_json_error( [ 'message' => 'Invalid base URL format.' ] );
    }

    $login_result = chatbot_plugin_knowledge_repository_login_with_cookie( $base_url, $username, $password );
    error_log( 'Knowledge repository login result: ' . print_r( $login_result, true ) );

    if ( isset( $login_result['error'] ) ) {
        wp_send_json_error( [ 'message' => $login_result['error'] ] );
        return;
    }

    $token = $login_result['data']['token'] ?? null;
    if ( ! $token ) {
        wp_send_json_error( [ 'message' => 'Login failed, token not returned.' ] );
        return;
    }

    update_option( 'chatbot_knowledge_repository_options', [
        'base_url' => $base_url,
        'username' => $username,
        'password' => $password,
        'token'    => $token,
    ] );

    wp_send_json_success( [ 'message' => 'Login successful!', 'token' => $token ] );
}

/* ═══════════════════════════════════════════════════════════════
   GET NOTEBOOKS
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_get_notebooks', 'chatbot_get_notebooks' );
function chatbot_get_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    $options  = get_option( 'chatbot_knowledge_repository_options', [] );
    $token    = $options['token']    ?? '';
    $base_url = $options['base_url'] ?? '';

    if ( ! $token || ! $base_url ) {
        wp_send_json_error( [ 'message' => 'Not logged in. Please login first.' ] );
        return;
    }

    $response = chatbot_plugin_get_notebooks( $base_url, $token );

    if ( isset( $response['status'] ) && $response['status'] == 401 ) {
        $new_token = chatbot_plugin_knowledge_repository_refresh_token();
        if ( ! $new_token ) {
            delete_option( 'chatbot_knowledge_repository_options' );
            wp_send_json_error( [ 'message' => 'Session expired. Please login again.' ] );
            return;
        }
        $response = chatbot_plugin_get_notebooks( $base_url, $new_token );
        if ( isset( $response['status'] ) && $response['status'] == 401 ) {
            delete_option( 'chatbot_knowledge_repository_options' );
            wp_send_json_error( [ 'message' => 'Authentication failed. Please login again.' ] );
            return;
        }
    }

    if ( isset( $response['error'] ) ) {
        wp_send_json_error( [ 'message' => $response['error'] ] );
        return;
    }

    wp_send_json_success( [
        'notebooks' => $response['notebooks'] ?? [],
        'message'   => 'Fetched notebooks successfully',
    ] );
}

/* ═══════════════════════════════════════════════════════════════
   GET ALL PUBLISHED PAGES / POSTS (for the add-page modal)
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_get_saved_page_based', 'chatbot_get_saved_page_based' );
function chatbot_get_saved_page_based() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    $pages  = get_posts( [
        'post_type'      => [ 'post', 'page' ],
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'orderby'        => 'title',
        'order'          => 'ASC',
    ] );
    $result = [];
    foreach ( $pages as $page ) {
        $result[] = [
            'id'      => $page->ID,
            'title'   => $page->post_title,
            'url'     => get_permalink( $page->ID ),
            'excerpt' => wp_trim_words(
                ! empty( $page->post_excerpt ) ? $page->post_excerpt : $page->post_content,
                20
            ),
        ];
    }

    wp_send_json_success( $result );
}

/* ═══════════════════════════════════════════════════════════════
   GET NOTEBOOKS FOR ONE PAGE
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_get_page_notebooks', 'chatbot_get_page_notebooks' );
function chatbot_get_page_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    $page_id = intval( $_POST['page_id'] ?? 0 );
    if ( ! $page_id ) {
        wp_send_json_error( [ 'message' => 'Invalid page ID' ] );
        return;
    }

    $data = get_option( 'chatbot_page_based_notebooks', [] );
    wp_send_json_success( [
        'notebooks'  => $data[ $page_id ]['notebooks'] ?? [],
        'page_title' => get_the_title( $page_id ),
    ] );
}

/* ═══════════════════════════════════════════════════════════════
   SAVE NOTEBOOKS FOR ONE PAGE  ← KEY FIX HERE
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_save_page_notebooks', 'chatbot_save_page_notebooks' );
function chatbot_save_page_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( [ 'message' => 'No permission' ] );
        return;
    }

    $page_id   = intval( $_POST['page_id'] ?? 0 );
    $notebooks = isset( $_POST['notebook_ids'] )
        ? array_values( array_unique( array_map( 'intval', (array) $_POST['notebook_ids'] ) ) )
        : [];

    if ( ! $page_id ) {
        wp_send_json_error( [ 'message' => 'Invalid page ID' ] );
        return;
    }

    $data = get_option( 'chatbot_page_based_notebooks', [] );

    /*
     * FIX: ALWAYS write the entry, even when notebook_ids is empty.
     *
     * The old code did:
     *   if ($notebooks) { save } else { unset($data[$page_id]); }
     *
     * That deleted the entire page record when all notebooks were unchecked,
     * so the card vanished on the next page reload.
     *
     * Deletion is only done explicitly via chatbot_remove_page_notebooks.
     */
    $data[ $page_id ] = [
        'title'     => get_the_title( $page_id ),
        'notebooks' => $notebooks,
    ];

    update_option( 'chatbot_page_based_notebooks', $data );
    wp_send_json_success( [ 'message' => 'Saved', 'page_id' => $page_id ] );
}

/* ═══════════════════════════════════════════════════════════════
   GET ALL PAGE → NOTEBOOKS MAP
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_get_all_page_notebooks', 'chatbot_get_all_page_notebooks' );
function chatbot_get_all_page_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );
    wp_send_json_success( get_option( 'chatbot_page_based_notebooks', [] ) );
}

/* ═══════════════════════════════════════════════════════════════
   REMOVE ONE PAGE  (explicit delete – only place unset is used)
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_remove_page_notebooks', 'chatbot_remove_page_notebooks' );
function chatbot_remove_page_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( [ 'message' => 'Access denied' ] );
        return;
    }

    $page_id = intval( $_POST['page_id'] ?? 0 );
    if ( ! $page_id ) {
        wp_send_json_error( [ 'message' => 'Invalid page ID' ] );
        return;
    }

    $data = get_option( 'chatbot_page_based_notebooks', [] );
    unset( $data[ $page_id ] );
    update_option( 'chatbot_page_based_notebooks', $data );

    wp_send_json_success( [ 'page_id' => $page_id ] );
}

/* ═══════════════════════════════════════════════════════════════
   REMOVE ALL PAGES
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_remove_all_page_notebooks', 'chatbot_remove_all_page_notebooks' );
function chatbot_remove_all_page_notebooks() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( [ 'message' => 'Access denied' ] );
        return;
    }

    update_option( 'chatbot_page_based_notebooks', [] );
    wp_send_json_success( [ 'message' => 'All page notebooks removed' ] );
}

/* ═══════════════════════════════════════════════════════════════
   LOGOUT
═══════════════════════════════════════════════════════════════ */
add_action( 'wp_ajax_chatbot_knowledge_repository_logout', 'chatbot_knowledge_repository_logout' );
function chatbot_knowledge_repository_logout() {
    check_ajax_referer( 'chatbot_login_nonce', 'nonce' );
    delete_option( 'chatbot_knowledge_repository_options' );
    wp_send_json_success( [ 'message' => 'You have been logged out successfully.' ] );
}

/* debug – remove in production */
error_log( 'Page notebooks: ' . json_encode( get_option( 'chatbot_page_based_notebooks' ) ) );