<?php
if (!defined('ABSPATH')) exit;


    function chatbot_load_config() {
        static $config = null;

        if ($config === null) {
            $config_path = dirname(__DIR__) . '/config.json';
            $config = [];
            if (file_exists($config_path)) {
                $config = json_decode(file_get_contents($config_path), true);
                error_log("Config loaded: " . print_r($config, true));
            } else {
                error_log("Config file not found at $config_path");
            }
        }

        return $config;
    }

    function chatbot_get_stripe_base_url() {
        $config = chatbot_load_config(); 
        $option_value = get_option('chatbot_stripe_base_url');
        $stripe_base_url = !empty($option_value) ? sanitize_text_field($option_value) : ($config['stripe_url'] ?? 'https://subscriptions.smartprocesses.cloud/api');

        error_log("chatbot_get_stripe_base_url: " . $stripe_base_url);
        return $stripe_base_url;
    }

    function chatbot_get_product_id() {
        $config = chatbot_load_config();
        $option_value = get_option('chatbot_stripe_product_id');
        $prod_id = !empty($option_value) ? sanitize_text_field($option_value) : ($config['prod_id'] ?? 'prod_TV5yFBE7tEW4LN');

        error_log("chatbot_get_product_id: " . $prod_id);
        return $prod_id;
    }

    function chatbot_check_and_log_subscription_status($subscription_data) {
        if (!is_array($subscription_data) || empty($subscription_data)) {
            return $subscription_data;
        }

        $current_status = isset($subscription_data['status']) ? (int)$subscription_data['status'] : null;
        $end_date = $subscription_data['currentPeriodEnd'] ?? null;

        error_log("Subscription Status from API: $current_status");

        if ($end_date) {
            try {
                $now_utc = new DateTime('now', new DateTimeZone('UTC'));
                $end_utc = new DateTime($end_date, new DateTimeZone('UTC'));

                error_log("Subscription Info - Now: " . $now_utc->format('Y-m-d H:i:s') . 
                        " | End: " . $end_utc->format('Y-m-d H:i:s') . 
                        " | Status from API: $current_status");

                if ($now_utc > $end_utc) {
                    error_log("Subscription expired but keeping status from API: $current_status");
                } else {
                    error_log("Subscription valid - Expires in: " . $now_utc->diff($end_utc)->format('%d days %h hours'));
                }
            } catch (Exception $e) {
                error_log("Error checking date: " . $e->getMessage());
            }
        }
        return $subscription_data;
    }

    add_action('wp_ajax_chatbot_add_trial', 'chatbot_add_trial_ajax');
        function chatbot_add_trial_ajax(){
            if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'chatbot_add_trial_nonce')) {
                error_log("Invalid nonce");
                wp_send_json_error(['message' => 'Invalid nonce']);
                wp_die();
            }

            $app_guid = get_option('chatbot_plugin_guid', '');
            error_log("app_guid: $app_guid");
            
            $stripe_base_url = chatbot_get_stripe_base_url();
            $product_id = get_option('chatbot_stripe_product_id','');

            if(empty($app_guid) || empty($stripe_base_url)){
                wp_send_json_error(['message' => 'Missing parameters']);
                wp_die();
            }
            $user_email = wp_get_current_user()->user_email;
            $title = get_option('siteurl');
            $add_trial_response = chatbot_stripe_add_trial($stripe_base_url, $app_guid, $product_id, $user_email, $title);
            
            update_option('chatbot_trial_response', $add_trial_response['subscription'] ?? []);
            error_log("Trial response: " . print_r($add_trial_response, true));

            if(isset($add_trial_response['error'])){
                wp_send_json_error(['message' => $add_trial_response['error']]);
            } else {
                wp_send_json_success($add_trial_response);
            }

            wp_die();
        }

    add_action('wp_enqueue_scripts', function() {
        wp_enqueue_script(
            'chatbot-frontend', 
            plugin_dir_url(__DIR__) . 'assets/js/chatbot-frontend.js', 
            ['jquery'], 
            '2.6.0', 
            true
        );
        
        $trial_response = get_option('chatbot_trial_response', []);
        $trial_response = chatbot_check_and_log_subscription_status($trial_response);

        $endDate = $trial_response['currentPeriodEnd'] ?? '';
        $status = $trial_response['status'] ?? '';
        $type = $trial_response['type'] ?? '';
        
        $stripe_base_url = chatbot_get_stripe_base_url();
        $app_guid = get_option('chatbot_plugin_guid', '');
        error_log('app guid base url for option' .$app_guid);
        $product_id = get_option('chatbot_stripe_product_id','');
        $user_email = wp_get_current_user()->user_email;
        error_log("Enqueue script - Trial End Date: " . $endDate);

        wp_localize_script('chatbot-frontend', 'chatbotData', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('chatbot_trial_nonce'),
            'trialEndDate' => $endDate,
            'stripe_base_url' => $stripe_base_url,
            'app_guid' => $app_guid,
            'product_id' => $product_id,
            'trial_status' => $status,
            'trial_type' => $type,
            'user_email' => $user_email ?? '',
        ]);
    });

    add_action('wp_ajax_chatbot_get_trial_status', 'chatbot_get_trial_status_ajax');
    add_action('wp_ajax_nopriv_chatbot_get_trial_status', 'chatbot_get_trial_status_ajax'); 
        function chatbot_get_trial_status_ajax() {
            $nonce = $_GET['nonce'] ?? $_REQUEST['nonce'] ?? '';
            
            if (!wp_verify_nonce($nonce, 'chatbot_trial_nonce')) {
                wp_send_json_error(['message' => 'Invalid nonce']);
                wp_die();
            }

            $app_guid = $_GET['app_guid'] ?? $_REQUEST['app_guid'] ?? get_option('chatbot_plugin_guid', '');
            $stripe_base_url = chatbot_get_stripe_base_url();
            
            error_log("Checking trial status for app_guid: $app_guid");
            
            $api_response = chatbot_stripe_active_subscription($stripe_base_url, $app_guid);
            error_log("API response after get active sub: " . json_encode($api_response));

            if(isset($api_response['error'])){
                error_log("API Error: " . $api_response['error']);
                wp_send_json_error(['message' => $api_response['error']]);
            } else {
                $api_response = chatbot_check_and_log_subscription_status($api_response);
                $trial_option = get_option('chatbot_trial_response', []);
                
                $oldEndDate = $trial_option['currentPeriodEnd'] ?? null;
                error_log("Old End Date session: " . $oldEndDate);
                $newEndDate = $api_response['currentPeriodEnd'] ?? null;
                error_log("New End Date api response: " . $newEndDate);
                $status = $api_response['status'] ?? null;
                $type = $api_response['type'] ?? null;
                $should_update = false;
                
                if ($newEndDate && $newEndDate !== $oldEndDate) {
                    $should_update = true;
                    error_log(" Date changed: $oldEndDate → $newEndDate");
                }
                
                if (isset($trial_option['status']) && $status != $trial_option['status']) {
                    $should_update = true;
                    error_log(" Status changed from API: " . $trial_option['status'] . " → $status");
                }
                
                if ($should_update) {
                    update_option('chatbot_trial_response', $api_response);
                    error_log("Updated trial response - End: $newEndDate, Status: $status");
                }
                
                if($api_response == null){
                    update_option('chatbot_trial_response', '');
                }

                wp_send_json_success([
                    'oldEndDate' => $oldEndDate,
                    'newEndDate' => $newEndDate,
                    'status' => $status,
                    'type' => $type,
                    'data' => $api_response
                ]);
            }

            wp_die();
        }

    add_action('wp_ajax_chatbot_get_all_subscriptions','chatbot_get_all_subscription_ajax');
    add_action('wp_ajax_nopriv_chatbot_get_all_subscriptions', 'chatbot_get_all_subscription_ajax');
        function chatbot_get_all_subscription_ajax(){
            $nonce = $_GET['nonce'] ?? $_REQUEST['nonce'];
            if(!wp_verify_nonce($nonce,'chatbot_trial_nonce')){
                wp_send_json_error(['message' =>'Invalid nonce']);
                wp_die();
            }

            $app_guid = get_option('chatbot_plugin_guid', '');
        
            $stripe_base_url = chatbot_get_stripe_base_url();
            error_log("stripe base url " .$stripe_base_url);
            $api_response = chatbot_get_all_subscriptions($stripe_base_url, $app_guid);
            
            if(isset($api_response['error'])){
            wp_send_json_error(['message' => $api_response['error']]);
            }else {
                if (is_array($api_response)) {
                    error_log(" Processing " . count($api_response) . " subscriptions");
                    foreach ($api_response as $key => $subscription) {
                        $api_response[$key] = chatbot_check_and_log_subscription_status($subscription);
                    }
                    if (!empty($api_response[0])) {
                        $current_trial = get_option('chatbot_trial_response', []);
                        if (isset($current_trial['id']) && 
                            isset($api_response[0]['id']) && 
                            $current_trial['id'] == $api_response[0]['id']) {
                            update_option('chatbot_trial_response', $api_response[0]);
                            error_log(" Updated chatbot_trial_response from subscriptions list");
                        }
                    }
                }
                wp_send_json_success($api_response); 
            }

            wp_die();
        }

    add_action('wp_ajax_chatbot_get_history_subscriptions','chatbot_get_history_subscription_ajax');
    add_action('wp_ajax_nopriv_chatbot_get_history_subscriptions', 'chatbot_get_history_subscriptions_ajax');
        function chatbot_get_history_subscription_ajax(){
            $nonce = $_GET['nonce'] ?? $_REQUEST['nonce'];
            if(!wp_verify_nonce($nonce,'chatbot_trial_nonce')){
                wp_send_json_error(['message' =>'Invalid nonce']);
                wp_die();
            }

            $app_guid = get_option('chatbot_plugin_guid', '');
        
            $stripe_base_url = chatbot_get_stripe_base_url();
            error_log("stripe base url " .$stripe_base_url);
            $api_response = chatbot_get_history_subscription($stripe_base_url,$app_guid);
            
            if(isset($api_response['error'])){
            wp_send_json_error(['message' => $api_response['error']]);
            }else {
            wp_send_json_success($api_response); 
            }
            wp_die();
        }


    add_action('wp_ajax_chatbot_cancel_subscription','chatbot_cancel_subscription_ajax');
        function chatbot_cancel_subscription_ajax(){
            $nonce = $_POST['nonce'] ?? '';
            if(!wp_verify_nonce($nonce,'chatbot_trial_nonce')){
                wp_send_json_error(['message' =>'Invalid nonce']);
                wp_die();
            }
            $subscription_id = $_POST['subscription_id'];
            $app_guid = get_option('chatbot_plugin_guid', '');
            $stripe_base_url = chatbot_get_stripe_base_url();
            $api_response = chatbot_plugin_cancel_subscription($stripe_base_url, $app_guid, $subscription_id, true);

            if(isset($api_response['error'])){
            wp_send_json_error(['message' => $api_response['error']]);
            } else {
            wp_send_json_success(['message' => 'Subscription  cancelled successfully.']); 
            }

            wp_die();
        }

    add_action('wp_ajax_chatbot_reactivate_subscription', 'chatbot_reactivate_subscription_ajax');
        function chatbot_reactivate_subscription_ajax(){
            $nonce = $_POST['nonce'] ?? '';
            if(!wp_verify_nonce($nonce,'chatbot_trial_nonce')){
                wp_send_json_error(['message' =>'Invalid nonce']);
                wp_die();
            }
            $subscription_id = $_POST['subscription_id'];
            $app_guid = get_option('chatbot_plugin_guid','');
            $stripe_base_url = chatbot_get_stripe_base_url();
            $product_id = get_option('chatbot_stripe_product_id');

            $api_response = chatbot_plugin_reactivate_subscription($stripe_base_url,$app_guid,$subscription_id,false);

            if(isset($api_response['error'])){
                wp_send_json_error(['message' => $api_response['error']]);
            } else {
                wp_send_json_success(['message' => 'Subscription reactivate successfully']);
            }
            wp_die();
        }

    add_action('wp_ajax_chatbot_add_trial_subscription', 'chatbot_add_trial_subscription_ajax');
        function chatbot_add_trial_subscription_ajax() {
            $nonce = $_POST['nonce'] ?? '';
            if (!wp_verify_nonce($nonce, 'chatbot_trial_nonce')) {
                wp_send_json_error(['message' => 'Invalid nonce']);
                wp_die();
            }

            $guid = $_POST['app_guid'] ?? '';
            $stripe_base_url = chatbot_get_stripe_base_url();
            $product_id =  chatbot_get_product_id();

            if (!is_user_logged_in()) {
                wp_send_json_error(['message' => 'User not logged in']);
                wp_die();
            }

            $user_email = wp_get_current_user()->user_email;
            $title = get_option('siteurl');

            $response = chatbot_stripe_add_trial($stripe_base_url, $guid, $product_id, $user_email, $title);
            if (!$response || isset($response['error'])) {
                wp_send_json_error(['message' => $response['error'] ?? 'Unknown error']);
                wp_die();
            }

            $subscription = $response['subscription'] ?? [];
            update_option('chatbot_trial_response', $subscription);
            
            if (isset($subscription['currentPeriodStart'])) 
                update_option('chatbot_trial_start', $subscription['currentPeriodStart']);
            if (isset($subscription['currentPeriodEnd']))
                update_option('chatbot_trial_end', $subscription['currentPeriodEnd']);
            if (isset($subscription['id'])) 
                update_option('chatbot_subscription_id', $subscription['id']);
            if (isset($subscription['customerEmail'])) 
                update_option('chatbot_customer_email', $subscription['customerEmail']);
            if (isset($subscription['status'])) 
                update_option('chatbot_subscription_status', $subscription['status']);

            wp_send_json_success($response);
            wp_die();
        }

    add_action('wp_ajax_chatbot_add_trial_without_user', 'chatbot_add_trial_without_user_ajax');
        function chatbot_add_trial_without_user_ajax() {
            $nonce = $_POST['nonce'] ?? '';
            if (!wp_verify_nonce($nonce, 'chatbot_trial_nonce')) {
                wp_send_json_error(['message' => 'Invalid nonce']);
                wp_die();
            }

            $guid = $_POST['app_guid'] ?? '';
            $stripe_base_url = chatbot_get_stripe_base_url();
            $product_id =  chatbot_get_product_id();

            if (!is_user_logged_in()) {
                wp_send_json_error(['message' => 'User not logged in']);
                wp_die();
            }

            $user_email = wp_get_current_user()->user_email;
            $title = get_option('siteurl');

            $response = chatbot_stripe_add_trial_withoutUser($stripe_base_url, $guid, $product_id, $user_email);
            if (!$response || isset($response['error'])) {
                wp_send_json_error(['message' => $response['error'] ?? 'Unknown error']);
                wp_die();
            }

            $subscription = $response['subscription'] ?? [];
            update_option('chatbot_trial_response', $subscription);
            
            if (isset($subscription['currentPeriodStart'])) 
                update_option('chatbot_trial_start', $subscription['currentPeriodStart']);
            if (isset($subscription['currentPeriodEnd']))
                update_option('chatbot_trial_end', $subscription['currentPeriodEnd']);
            if (isset($subscription['id'])) 
                update_option('chatbot_subscription_id', $subscription['id']);
            if (isset($subscription['customerEmail'])) 
                update_option('chatbot_customer_email', $subscription['customerEmail']);
            if (isset($subscription['status'])) 
                update_option('chatbot_subscription_status', $subscription['status']);

            wp_send_json_success($response);
            wp_die();
        }

add_action('wp_ajax_chatbot_download_invoice', 'chatbot_download_invoice');
function chatbot_download_invoice() {
    $nonce = $_POST['nonce'] ?? '';
    if (!wp_verify_nonce($nonce, 'chatbot_trial_nonce')) {
        wp_send_json_error(['message' => 'Invalid nonce']);
        wp_die();
    }

    $stripe_base_url = chatbot_get_stripe_base_url();
    $invoice_number = sanitize_text_field($_POST['invoice_number'] ?? '');
    if (empty($invoice_number)) {
        wp_send_json_error(['message' => 'Invoice number missing']);
        wp_die();
    }

    $result = chatbot_downloadInvoice($stripe_base_url, $invoice_number);

    if (isset($result['url'])) {
        wp_send_json_success(['url' => $result['url']]);
    } else {
        wp_send_json_error(['message' => $result['error'] ?? 'Unknown error']);
    }

    wp_die();
}
